
/*
 * 2004 Jan Struyf
 *
 */

#include "../basicconf.h"
#include "../gprint.h"
#include "img2ps.h"

GLEBitmap::GLEBitmap() {
	m_Height = 0;
	m_Width = 0;
	m_Mode = GLE_BITMAP_NONE;
	m_Components = 1;
	m_ExtraComponents = 0;
	m_Colors = 0;
	m_Interlaced = 0;
	m_Palette = NULL;
	m_ASCII85 = 0;
	m_Compress = 0;
	m_Alpha = 0;
	m_Encoding = GLE_BITMAP_LZW;
	m_BitsPerComponent = 8;
}

GLEBitmap::~GLEBitmap() {
	close();
	if (m_Palette != NULL) delete m_Palette;
}

int GLEBitmap::open(const string& fname) {
	return 0;
}

int GLEBitmap::readHeader() {
	return 0;
}

int GLEBitmap::toPS(FILE* fp) {
	// Init image, set transformations, possibly load palette
	prepare(GLE_BITMAP_PREPARE_SCANLINE);
	// Variables
	const char* coder = "/ASCII85Decode filter";
	const char* encoder;
	int width = m_Width;
	int height = m_Height;
	int colors = getNbColors();
	int bits = getBitsPerComponent();
	if (getEncoding() == GLE_BITMAP_LZW) {
		encoder = "/LZWDecode";
	} else {
		encoder = "/DCTDecode";
	}
	fprintf(fp, "save 9 dict begin\n");
	fprintf(fp, "{/T currentfile%s def\n", coder);
	if (isIndexed()) {
		fprintf(fp, "[/Indexed/DeviceRGB %d T %d string readstring pop]", colors-1, colors*3);
	} else if (isGrayScale()) {
		fprintf(fp, "/DeviceGray");
	} else {
		fprintf(fp, "/DeviceRGB");
	}
	fprintf(fp, " setcolorspace\n");
	fprintf(fp, "/F T%s filter def\n", encoder);
	fprintf(fp, "<</ImageType 1/Width %d/Height %d/BitsPerComponent %d\n", width, height, bits);
	fprintf(fp, "/ImageMatrix[%d 0 0 -%d 0 %d]/Decode\n", width, height, height);
	fprintf(fp, "[");
	int decodemax = isIndexed() ? (1 << bits)-1 : 1;
	fprintf(fp, "0 %d", decodemax);
	int comps = getColorComponents();
	for (int j = 1; j < comps; j++) {
		fprintf(fp, " 0 %d", decodemax);
	}
	fprintf(fp, "]/DataSource F>> image\n");
	fprintf(fp, "F closefile T closefile}\n");
	fprintf(fp, "exec\n");
	GLEASCII85ByteStream ascii85(fp);
	// Store palette for indexed image
	if (isIndexed()) {
		rgb* pal = getPalette();
		for (int i = 0; i < colors; i++) {
			ascii85.sendByte(pal[i].red);
			ascii85.sendByte(pal[i].green);
			ascii85.sendByte(pal[i].blue);
		}
	}
	// Encode image
	GLEByteStream* stream = NULL;
	if (getEncoding() == GLE_BITMAP_LZW) {
		GLELZWByteStream lzw(&ascii85);
		// Remove extra components?
		int extra = getExtraComponents();
		int color_alpha = getColorComponents();
		if (isAlpha()) {
			// Do not count alpha as extra component
			extra--;
			color_alpha++;
		}
		GLEComponentRemovalByteStream crem(&lzw, color_alpha, extra);
		if (extra != 0) {
			stream = &crem;
		} else {
			stream = &lzw;
		}
		// Remove alpha?
		GLEAlphaRemovalByteStream alpha(stream, color_alpha);
		if (isAlpha()) stream = &alpha;
		// Combine more than one pixel in each byte?
		GLEPixelCombineByteStream combine(stream, bits);
		if (bits < 8) stream = &combine;
		// Decode
		decode(stream);
		stream->term();
	} else {
		coded(&ascii85);
	}
	// End
	ascii85.term();
	fprintf(fp, "end restore\n");
	return 0;
}

int GLEBitmap::decode(GLEByteStream* output) {
	return GLE_IMAGE_ERROR_NOT_IMPL;
}

int GLEBitmap::coded(GLEByteStream* output) {
	return GLE_IMAGE_ERROR_NOT_IMPL;
}

void GLEBitmap::loadImageData() {
}

int GLEBitmap::prepare(int mode) {
	return GLE_IMAGE_ERROR_NONE;
}

rgb* GLEBitmap::allocPalette(int ncolors) {
	if (m_Palette != NULL) delete m_Palette;
	m_Palette = new rgb[ncolors];
	return m_Palette;
}

int GLEBitmap::getScanlineSize() {
	int bytespc = getBitsPerComponent()/8;
	if (bytespc < 1) bytespc = 1;
	return getWidth()*getComponents()*bytespc;
}

int GLEBitmap::getColorComponents() {
	return getComponents() - getExtraComponents();
}

int GLEBitmap::getMaxBits() {
	if (isIndexed()) {
		if (m_Colors > 16) {
			/* More than 16 colors -> 8 bits (5,6,7 bits are not supported by postscript!) */
			return 8;
		} else if (m_Colors >= 5) {
			/* At most 16 colors */
			return 4;
		} else if (m_Colors >= 3) {
			/* At most 4 colors */
			return 2;
		} else {
			/* Monochrome bitmap */
			return 1;
		}
	} else {
		return 8;
	}
}

void GLEBitmap::checkGrayScalePalette() {
	rgb* pal = getPalette();
	if (getNbColors() == 256) {
		int nogray = 0;
		for (int i = 0; i < 256; i++) {
			if (pal[i].red != i || pal[i].green != i || pal[i].blue != i) {
				nogray = 1;
			}
		}
		if (nogray == 0) {
			setMode(GLE_BITMAP_GRAYSCALE);
			setBitsPerComponent(8);
		}
	} else if (getNbColors() == 2) {
		if (pal[0].red == 0   && pal[0].green == 0   && pal[0].blue == 0 &&
		    pal[1].red == 255 && pal[1].green == 255 && pal[1].blue == 255) {
			setMode(GLE_BITMAP_GRAYSCALE);
			setBitsPerComponent(1);
		}
	}
}

void GLEBitmap::printInfo(ostream& os) {
	os << getWidth();
	os << "x";
	os << getHeight();
	os << "x";
	os << getBitsPerComponent() * getComponents();
	switch (getMode()) {
		case GLE_BITMAP_GRAYSCALE:
			os << "-GRAY";
			break;
		case GLE_BITMAP_RGB:
			os << "-RGB";
			break;
		case GLE_BITMAP_INDEXED:
			os << "-PAL:" << getNbColors();
			break;
	}
}

void GLEBitmap::close() {
}

GLEFileBitmap::GLEFileBitmap() {
	m_In = NULL;
	m_Palette = NULL;
}

int GLEFileBitmap::open(const string& fname) {
	setFName(fname);
	m_In = fopen(fname.c_str(), "rb");
	if (m_In == NULL) {
		return 0;
	} else {
		return 1;
	}
}

int GLEFileBitmap::read16BE() {
	int lh = fgetc(m_In);
	int ll = fgetc(m_In);
	return (lh << 8) | ll;
}

int GLEFileBitmap::read16LE() {
	int lh = fgetc(m_In);
	int ll = fgetc(m_In);
	return (ll << 8) | lh;
}

void GLEFileBitmap::close() {
	if (m_In != NULL) {
		fclose(m_In);
		m_In = NULL;
	}
}

GLEByteStream::GLEByteStream() {
	m_Terminated = 0;
}

GLEByteStream::~GLEByteStream() {
}

int GLEByteStream::send(GLEBYTE* bytes, GLEDWORD count) {
	for (int i = 0; i < count; i++) {
		sendByte(bytes[i]);
	}
	return GLE_IMAGE_ERROR_NONE;
}

int GLEByteStream::term() {
	m_Terminated = 1;
	return GLE_IMAGE_ERROR_NONE;
}

int GLEByteStream::endScanLine() {
	return GLE_IMAGE_ERROR_NONE;
}

GLEPipedByteStream::GLEPipedByteStream(GLEByteStream* pipe): GLEByteStream() {
	m_Pipe = pipe;
}

GLEPipedByteStream::~GLEPipedByteStream() {
}

int GLEPipedByteStream::endScanLine() {
	m_Pipe->endScanLine();
	return GLEByteStream::endScanLine();
}

int GLEPipedByteStream::term() {
	m_Pipe->term();
	return GLEByteStream::term();
}

GLEPixelCombineByteStream::GLEPixelCombineByteStream(GLEByteStream* pipe, int bpc) : GLEPipedByteStream(pipe) {
	m_BitsPerComponent = bpc;
	m_BitsLeft = 8;
	m_Combined = 0;
}

GLEPixelCombineByteStream::~GLEPixelCombineByteStream() {
}

int GLEPixelCombineByteStream::flushBufferByte() {
	m_Pipe->sendByte(m_Combined);
	m_BitsLeft = 8;
	m_Combined = 0;
	return GLE_IMAGE_ERROR_NONE;
}

int GLEPixelCombineByteStream::sendByte(GLEBYTE byte) {
	if (m_BitsLeft >= m_BitsPerComponent) {
		/* Still fits in */
		m_Combined |= byte << (m_BitsLeft - m_BitsPerComponent);
		m_BitsLeft -= m_BitsPerComponent;
	} else {
		/* Does not fit in */
		int left = m_BitsPerComponent - m_BitsLeft;
		m_Combined |= byte >> left;
		flushBufferByte();
		m_Combined |= byte << (m_BitsLeft - left);
		m_BitsLeft -= left;
	}
	if (m_BitsLeft == 0) flushBufferByte();
	return GLE_IMAGE_ERROR_NONE;
}

int GLEPixelCombineByteStream::endScanLine() {
	if (m_BitsLeft != 8) flushBufferByte();
	return GLEPipedByteStream::endScanLine();
}

int GLEPixelCombineByteStream::term() {
	if (m_BitsLeft != 8) flushBufferByte();
	return GLEPipedByteStream::term();
}

GLEComponentRemovalByteStream::GLEComponentRemovalByteStream(GLEByteStream* pipe, int main, int remove) : GLEPipedByteStream(pipe) {
	m_Index = 0;
	m_Removed = 0;
	m_Main = main;
	m_Total = main+remove;
}

GLEComponentRemovalByteStream::~GLEComponentRemovalByteStream() {
}

int GLEComponentRemovalByteStream::sendByte(GLEBYTE byte) {
	if (m_Index < m_Main) {
		m_Pipe->sendByte(byte);
	} else {
		m_Removed++;
	}
	m_Index++;
	if (m_Index >= m_Total) {
		m_Index = 0;
	}
	return GLE_IMAGE_ERROR_NONE;
}

int GLEComponentRemovalByteStream::endScanLine() {
	m_Index = 0;
	return GLEPipedByteStream::endScanLine();
}

GLEAlphaRemovalByteStream::GLEAlphaRemovalByteStream(GLEByteStream* pipe, int components) : GLEPipedByteStream(pipe) {
	m_Components = components-1;
	m_Index = 0;
	// Do not support images with more channels
	if (m_Components > GLE_BITMAP_MAX_COMPONENTS) {
		m_Components = GLE_BITMAP_MAX_COMPONENTS;
	}
}

GLEAlphaRemovalByteStream::~GLEAlphaRemovalByteStream() {
}

int GLEAlphaRemovalByteStream::sendByte(GLEBYTE byte) {
	if (m_Index >= m_Components) {
		unsigned int adjust = 255-byte;
		for (int i = 0; i < m_Components; i++) {
			unsigned int output = (unsigned int)m_Buffer[i] + adjust;
			if (output >= 0xFF) m_Pipe->sendByte(0xFF);
			else m_Pipe->sendByte((GLEBYTE)output);
		}
		m_Index = 0;
	} else {
		m_Buffer[m_Index++] = byte;
	}
	return GLE_IMAGE_ERROR_NONE;
}

int GLEAlphaRemovalByteStream::endScanLine() {
	m_Index = 0;
	return GLEPipedByteStream::endScanLine();
}

BicubicIpolDoubleMatrix::BicubicIpolDoubleMatrix(double* data, int wd, int hi) {
	m_Width = wd;
	m_Height = hi;
	m_Data = data;
}

double BicubicIpolDoubleMatrix::getValue(int x, int y) {
	return m_Data[y*m_Width + x];
}

BicubicIpol::BicubicIpol(BicubicIpolData* data, int sw, int sh) {
	m_Data = data;
	m_SWidth = sw;
	m_SHeight = sh;
	m_Width = data->getWidth();
	m_Height = data->getHeight();
	m_SX = (double)m_Width / m_SWidth;
	m_SY = (double)m_Height / m_SHeight;
}

double BicubicIpol::R(double x) {
	double v;
	double sum = 0.0;
	v = x + 2;
	if (v > 0) sum += v*v*v;
	v = x + 1;
	if (v > 0) sum -= 4*v*v*v;
	if (x > 0) sum += 6*x*x*x;
	v = x - 1;
	if (v > 0) sum -= 4*v*v*v;
	return sum/6.0;
}

double BicubicIpol::ipol(int xp, int yp) {
	double x = xp * m_SX;
	double y = yp * m_SY;
	int i = (int)floor(x);
	int j = (int)floor(y);
	double dx = x - i;
	double dy = y - j;
	double value = 0;
	for (int m = -1; m <= 2; m++) {
		int xo = i + m;
		double rx = R(m-dx);
		for (int n = -1; n <= 2; n++) {
			int yo = j + n;
			if (xo < 0) xo = 0;
			if (xo >= m_Width) xo = m_Width-1;
			if (yo < 0) yo = 0;
			if (yo >= m_Height) yo = m_Height-1;
			value += m_Data->getValue(xo, yo) * rx * R(dy-n);
		}
	}
	return value;
}

void GLEBitmapSetPalette(GLEBYTE* pal, int offs, double red, double green, double blue) {
	int i_red = (int)floor(red*255+0.5);
	int i_green = (int)floor(green*255+0.5);
	int i_blue = (int)floor(blue*255+0.5);
	if (i_red > 255) i_red = 255;
	if (i_green > 255) i_green = 255;
	if (i_blue > 255) i_blue = 255;
	if (i_red < 0) i_red = 0;
	if (i_green < 0) i_green = 0;
	if (i_blue < 0) i_blue = 0;
	pal[offs*3] = i_red;
	pal[offs*3+1] = i_green;
	pal[offs*3+2] = i_blue;
}

GLEBYTE* GLEBitmapCreateColorPalette(int nc) {
	int offset = 1;
	int nintervals = 6;
	int divisor = 3;
	int num = nc-offset;
	int n = (num / (nintervals*divisor))*divisor;
	nc = n*nintervals+offset;
	GLEBYTE* result = new GLEBYTE[nc*3];
	double ninv = 1.0/n;
	int n3 = n/3;
	int n23 = 2*n3;
	double third = n3*ninv;
	double twothirds = n23*ninv;
	for (int i = 0; i < n3; ++i) {
		double ininv = i*ninv;
		GLEBitmapSetPalette(result, i, ininv, 0.0, ininv);
		GLEBitmapSetPalette(result, n3+i, third, 0.0, third+ininv);
		GLEBitmapSetPalette(result, n23+i, third-ininv, 0.0, twothirds+ininv);
	}
	for (int i = 0; i < n; ++i) {
		double ininv = i*ninv;
		double ininv1 = 1.0-ininv;
		GLEBitmapSetPalette(result, n+i, 0.0, ininv, 1.0);
		GLEBitmapSetPalette(result, 2*n+i, 0.0, 1.0, ininv1);
		GLEBitmapSetPalette(result, 3*n+i, ininv, 1.0, 0.0);
		GLEBitmapSetPalette(result, 4*n+i, 1.0, ininv1, 0.0);
		GLEBitmapSetPalette(result, 5*n+i, 1.0, ininv, ininv);
	}
	GLEBitmapSetPalette(result, 6*n, 1.0, 1.0, 1.0);
	return result;
}
