/*
 * 2004 Jan Struyf
 *
 */

#include "all.h"
#include "SourceLine.h"
#include "tokens/Tokenizer.h"
#include "tokens/stokenizer.h"
#include "mem_limits.h"
#include "token.h"
#include "core.h"
#include "mygraph.h"
#include "file_io.h"
#include "texinterface.h"
#include "cutils.h"
#include "cmdline.h"
#include "gprint.h"
#include "config.h"
#include "drawit.h"

#ifdef __WIN32__
	#include <windows.h>
#endif

#define BEGINDEF extern
#include "begin.h"

extern string GLE_TOP_DIR;
extern string GLE_BIN_DIR;
extern _GLESource GLESource;
extern ConfigCollection g_Config;
extern CmdLineObj g_CmdLine;

void doskip(char *s,int *ct);
char *un_quote(char *ct);

#define skipspace doskip(tk[ct],&ct)

void begin_config(const char* block, int *pln, int *pcode, int *cp) {
	string block_name(block);
	ConfigSection* section = g_Config.getSection(block_name);
	if (section == NULL) {
		gprint("Unrecognised CONFIG section {%s}\n ", block_name.c_str());
	}
	// Start with pcode from the next line
	(*pln)++;
	begin_init();
	while (true) {
		int st = begin_token(&pcode,cp,pln,srclin,tk,&ntk,outbuff);
		if (!st) {
			/* exit loop */
			break;
		}
		int ct = 1;
		int mode = 0;
		bool plus_is = false;
		CmdLineOption* option = NULL;
		while (ct <= ntk) {
			skipspace;
			if (section != NULL) {
				if (mode == 0) {
					option = section->getOption(tk[ct]);
					if (option == NULL) {
						gprint("Not a valid setting for section '%s': {%s}\n", block_name.c_str(), tk[ct]);
					}
				} else if (mode == 1) {
					if (strcmp(tk[ct], "=") == 0) {
						plus_is = false;
					} else if (strcmp(tk[ct], "+=") == 0) {
						plus_is = true;
					} else {
						gprint("Expected '=' or '+=', not {%s}\n", tk[ct]);
					}
				} else if (option != NULL) {
					CmdLineOptionArg* arg = option->getArg(0);
					if (!plus_is) arg->reset();
					arg->appendValue(tk[ct]);
				}
				mode++;
			}
			ct++;
		}
	}
}

void init_config(ConfigCollection* collection) {
	ConfigSection* section;
	CmdLineOption* option;
	CmdLineArgString* strarg;
	CmdLineArgSet* setarg;
	CmdLineArgSPairList* sparg;
/* GLE */
	section = new ConfigSection("gle");
	strarg = section->addStringOption("current", GLE_CONFIG_GLE_VERSION);
	strarg->setDefault("");
	section->addSPairListOption("versions", GLE_CONFIG_GLE_INSTALL);
	collection->addSection(section, GLE_CONFIG_GLE);
/* Tools */
	section = new ConfigSection("tools");
	/* LaTeX */
	strarg = section->addStringOption("tex", GLE_TOOL_LATEX_CMD);
#ifdef __WIN32__
	strarg->setDefault("latex.exe");
#endif
#ifdef __UNIX__
	strarg->setDefault("latex");
#endif
#ifdef __OS2__
	strarg->setDefault("vlatexp.cmd");
#endif
	/* PdfLaTeX */
	strarg = section->addStringOption("pdftex", GLE_TOOL_PDFTEX_CMD);
#ifdef __WIN32__
	strarg->setDefault("pdflatex.exe");
#else
	strarg->setDefault("pdflatex");
#endif
	/* DVIPS */
	strarg = section->addStringOption("dvips", GLE_TOOL_DVIPS_CMD);
#ifdef __WIN32__
	strarg->setDefault("dvips.exe");
#endif
#ifdef __UNIX__
	strarg->setDefault("dvips");
#endif
#ifdef __OS2__
	strarg->setDefault("dvips.exe");
#endif
	/* GhostScript */
	strarg = section->addStringOption("ghostscript", GLE_TOOL_GHOSTSCRIPT_CMD);
#ifdef __WIN32__
	strarg->setDefault("gswin32c.exe");
#endif
#ifdef __UNIX__
	strarg->setDefault("gs");
#endif
#ifdef __OS2__
	strarg->setDefault("gsos2.exe");
#endif
	collection->addSection(section, GLE_CONFIG_TOOLS);
/* TeX config */
	section = new ConfigSection("tex");
	option = new CmdLineOption("system");
	setarg = new CmdLineArgSet("device-names");
	setarg->setMaxCard(1);
	setarg->addPossibleValue("latex");
	setarg->addPossibleValue("vtex");
#ifdef __OS2__
	setarg->addDefaultValue(GLE_TEX_SYSTEM_VTEX);
#else
	setarg->addDefaultValue(GLE_TEX_SYSTEM_LATEX);
#endif
	option->addArg(setarg);
	section->addOption(option, GLE_TEX_SYSTEM);
	collection->addSection(section, GLE_CONFIG_TEX);
/* Config paper */
	section = new ConfigSection("paper");
	strarg = section->addStringOption("size", GLE_CONFIG_PAPER_SIZE);
	strarg->setDefault("a4paper");
	strarg = section->addStringOption("margins", GLE_CONFIG_PAPER_MARGINS);
	strarg->setDefault("2.54 2.54 2.54 2.54");
	collection->addSection(section, GLE_CONFIG_PAPER);
	collection->setDefaultValues();
}

const string& gle_config_margins() {
	ConfigSection* paper = g_Config.getSection(GLE_CONFIG_PAPER);
	return ((CmdLineArgString*)paper->getOptionValue(GLE_CONFIG_PAPER_MARGINS))->getValue();
}

const string& gle_config_papersize() {
	ConfigSection* paper = g_Config.getSection(GLE_CONFIG_PAPER);
	return ((CmdLineArgString*)paper->getOptionValue(GLE_CONFIG_PAPER_SIZE))->getValue();
}

#ifdef __WIN32__
bool gle_load_registry(const char* gle_key, string* gle_exe) {
	HKEY hk;
	if (RegOpenKeyEx(HKEY_CURRENT_USER, gle_key, 0, KEY_READ, &hk) == ERROR_SUCCESS) {
		unsigned char buffer[1024];
		DWORD dwtype, dwbsize = 1024;
		if (RegQueryValueEx(hk, "GLE_EXE", NULL, &dwtype, buffer, &dwbsize) == ERROR_SUCCESS) {
			*gle_exe = (char*)buffer;
		}
		RegCloseKey(hk);
		return true;
	} else {
		return false;
	}
}

void gle_save_registry(const char* gle_key, const string& gle_exe) {
	HKEY hk;
	DWORD dwDisp;
	if (RegCreateKeyEx(HKEY_CURRENT_USER, gle_key, 0, NULL, REG_OPTION_NON_VOLATILE, KEY_WRITE, NULL, &hk, &dwDisp)) {
		printf("Could not create a registry key for GLE.");
		return;
	}
	const char* val = gle_exe.c_str();
	RegSetValueEx(hk, "GLE_EXE", 0, REG_SZ, (const BYTE*)val, strlen(val)+1);
	RegCloseKey(hk);
}
#endif

bool try_load_config(const string& fname) {
	if (text_load(fname, GLESource)) {
		g_select_device(GLE_DEVICE_DUMMY);
		g_message_first_newline(false);
		DrawIt(fname, GLESource, NULL, true);
		GLESource.clear();
		return true;
	} else {
		return false;
	}
}

bool try_save_config(const string& fname, ConfigCollection* collection) {
	if (collection->allDefaults()) {
		cout << "Collection::All defaults" << endl;
		return true;
	}
	ofstream fout(fname.c_str());
	if (!fout.is_open()) {
		cout << ">>> Can't write to config file '" << fname << "'" << endl;
		return false;
	}
	for (int i = 0; i < collection->getNbSections(); i++) {
		ConfigSection* sec = collection->getSection(i);
		if (!sec->allDefaults()) {
			fout << "begin config " << sec->getName() << endl;
			for (int j = 0; j < sec->getNbOptions(); j++) {
				CmdLineOption* option = sec->getOption(j);
				if (!option->allDefaults()) {
					fout << "\t" << option->getName() << " = ";
					for (int k = 0; k < option->getMaxNbArgs(); k++) {
						if (k != 0) fout << " ";
						CmdLineOptionArg* arg = option->getArg(k);
						arg->write(fout);
					}
					fout << endl;
				}
			}
			fout << "end config" << endl << endl;
		}
	}
	fout.close();
	return true;
}

void get_version_soft(const string& cmdline, string& version) {
	// Running GLE gives something like: "GLE version 4.0.11"
	// Parse this string to get version number
	string gle_output;
	GLERun(cmdline, gle_output);
	str_parse_get_next(gle_output, "VERSION", version);
}

void get_version_hard(const string& cmdline, string& version) {
	// If gle_version_soft fails, use this more complex method
	// Running GLE on some dummy file results in "GLE 4.0.11 [temp1234.gle]-C-R-[temp1234.eps]"
	// Parse this string to get version number
	string gle_output;
	string temp_file = "temp1234";
	// Set GLE top
	GLESetGLETop(cmdline);
	// Create dummy file
	string temp_gle_file = temp_file + ".gle";
	ofstream file(temp_gle_file.c_str());
	file << "size 10 10" << endl;
	file << "amove 0 0" << endl;
	file.close();
	// Create command line
	string cmd_line = string("\"") + cmdline + "\" " + temp_gle_file;
	GLERun(cmd_line, gle_output);
	// Parse output to find version number
	str_parse_get_next(gle_output, "GLE", version);
	// Delete our temp file
	TryDeleteFile(temp_gle_file);
	TryDeleteFile(temp_file + ".ps");
}

void init_installed_versions(CmdLineObj& cmdline, ConfigCollection* collection) {
	CmdLineArgSet* versions = (CmdLineArgSet*)cmdline.getOption(GLE_OPT_VERSION)->getArg(0);
	ConfigSection* gle = collection->getSection(GLE_CONFIG_GLE);
	CmdLineArgSPairList* installs = (CmdLineArgSPairList*)gle->getOption(GLE_CONFIG_GLE_INSTALL)->getArg(0);
	for (int i = 0; i < installs->size(); i++) {
		versions->addPossibleValue(installs->getValue1(i).c_str());
	}
}

void wait_for_enter() {
	char Line[5];
	fgets(Line, sizeof(Line), stdin);
}

void do_wait_for_enter() {
	if (g_CmdLine.hasOption(GLE_OPT_PAUSE)) {
		cout << "Press enter to continue ..." << endl;
		wait_for_enter();
	}
}

void do_wait_for_enter_exit(int exitcode) {
	do_wait_for_enter();
	exit(exitcode);
}

void find_deps(const string& loc, ConfigCollection* collection) {
	vector<string> tofind;
	vector<string*> result;
	string gle_paths = ";";
#ifdef __WIN32__
	tofind.push_back("gle.exe");
	result.push_back(&gle_paths);
	tofind.push_back("gle_ps.exe");
	result.push_back(&gle_paths);
#endif
	ConfigSection* tools = collection->getSection(GLE_CONFIG_TOOLS);
	for (int i = 0; i < tools->getNbOptions(); i++) {
		CmdLineArgString* strarg = (CmdLineArgString*)tools->getOption(i)->getArg(0);
		tofind.push_back(strarg->getDefault());
		result.push_back(strarg->getValuePtr());
	}
	cout << "Running GLE -finddeps \"" << loc << "\" (locate GLE fonts and optionally Ghostscript/LaTeX): "; fflush(stdout);
	GLEFindFiles(loc, tofind, result);
	cout << endl;
	// Write installed GLE's to config section
	ConfigSection* gle = collection->getSection(GLE_CONFIG_GLE);
	CmdLineArgSPairList* installs = (CmdLineArgSPairList*)gle->getOption(GLE_CONFIG_GLE_INSTALL)->getArg(0);
	char_separator separator(";", "");
	tokenizer<char_separator> tokens(gle_paths, separator);
	while (tokens.has_more()) {
		string path = tokens.next_token();
		if (path.length() > 0 && !installs->hasValue2(path)) {
			installs->addPair("?", path);
		}
	}
	// Find versions of installed GLEs and set value of gleexe
	string gle_version = __GLEVN__;
	if (installs->size() > 1) {
		// Only need to find out versions if more than one installed
		// otherwise assume it is "this" version
		for (int i = 0; i < installs->size(); i++) {
			const string& cr_gle = installs->getValue2(i);
			string& version = installs->getValue1(i);
			if (version == "?") {
				get_version_soft(cr_gle, version);
				if (version == "?") {
					// cout << "Use hard method for: " << cr_gle << endl;
					get_version_hard(cr_gle, version);
				}
			}
			if (str_i_equals(version, gle_version)) {
				cout << "Found: GLE " << version << " in " << cr_gle << " (*)" << endl;
			} else {
				cout << "Found: GLE " << version << " in " << cr_gle << endl;
			}
		}
	} else if (installs->size() == 1) {
		cout << "Found: GLE in " << installs->getValue2(0) << endl;
		// Do not need to remember installed GLEs if there is only one
		// because then the "-v" option makes no sense
		installs->reset();
	} else {
		cout << "Found: GLE in ?" << endl;
	}
	// Show locations of other tools
	for (int i = 2; i < result.size(); i++) {
		CmdLineArgString* strarg = (CmdLineArgString*)tools->getOption(i-2)->getArg(0);
		if (result[i]->length() == 0 || strarg->isDefault()) {
			cout << "Found: " << tofind[i] << " in '?'" << endl;
		} else {
			cout << "Found: " << tofind[i] << " in '" << *(result[i]) << "'" << endl;
		}
	}
	do_wait_for_enter();
}

void complain_about_gletop(bool has_top) {
	if (has_top) {
		// If the user has GLE_TOP, then it points to the wrong location
		// Let him first try to remove it, to see if that works
		cerr << "GLE_TOP might be pointing to an incorrect location." << endl;
		cerr << "Try removing GLE_TOP from your environment." << endl;
	} else {
		// The user did not set GLE_TOP and still something is wrong
		// Ask him to correct this by setting GLE_TOP as a workaround
		cerr << "Please set GLE_TOP to the correct location." << endl;
	}
}

void check_correct_version(const string name, bool has_top, bool has_config, ConfigCollection& coll) {
	if (has_config) {
		const string& val = coll.getStringValue(GLE_CONFIG_GLE, GLE_CONFIG_GLE_VERSION);
		if (!str_i_equals(val.c_str(), __GLEVN__)) {
			cerr << "Error: GLE's configuration file:" << endl;
			cerr << "       '" << name << "'" << endl;
			cerr << "Is from GLE version '";
			if (val == "") { cerr << "unknown"; } else { cerr << val; }
			cerr << "' (and not '" << __GLEVN__ << "' as espected)." << endl;
			complain_about_gletop(has_top);
			exit(-1);
		}
	} else {
		cerr << "Error: GLE is unable to locate its configuration file:" << endl;
		cerr << "       '" << name << "'" << endl;
		complain_about_gletop(has_top);
		exit(-1);
	}
	coll.setStringValue(GLE_CONFIG_GLE, GLE_CONFIG_GLE_VERSION, __GLEVN__);
}

bool do_load_config(CmdLineObj& cmdline, ConfigCollection& coll) {
	// Set GLE_TOP
	// -> prefer environment var GLE_TOP
	// -> otherwise, locate relative to executable location
	string conf_name;
	bool has_top = false;
	bool has_config = false;
	const char* top = getenv("GLE_TOP");
	if (top == NULL || top[0] == 0) {
		string exe_name;
		bool has_exe_name = GetExeName("gle", exe_name);
		if (has_exe_name) {
			GetDirName(exe_name, GLE_BIN_DIR);
			AddDirSep(GLE_BIN_DIR);
			#ifdef GLETOP_CD
				GLE_TOP_DIR = exe_name;
				StripPathComponents(&GLE_TOP_DIR, GLETOP_CD+1);
				AddDirSep(GLE_TOP_DIR);
				GLE_TOP_DIR += GLETOP_REL;
				StripDirSep(GLE_TOP_DIR);
				conf_name = GLE_TOP_DIR + DIR_SEP + "glerc";
				has_config = try_load_config(conf_name);
				if (!has_config) {
					GLE_TOP_DIR = exe_name;
					StripPathComponents(&GLE_TOP_DIR, 2);
				}
			#else
				GLE_TOP_DIR = exe_name;
				StripPathComponents(&GLE_TOP_DIR, 2);
			#endif
		} else {
			// The user will see as error messege: "$GLE_TOP/some_file" not found.
			GLE_TOP_DIR = "$GLE_TOP";
		}
	} else {
		has_top = true;
		GLE_TOP_DIR = top;
	}
	StripDirSep(GLE_TOP_DIR);
	if (!has_config) {
		// Try load config file
		// -> first $GLE_TOP/glerc
		// -> on Unix also $HOME/.glerc
		if (conf_name == "") {
			// Only update conf_name if it was not yet set
			// To make error message if glerc not found more interpretable
			conf_name = GLE_TOP_DIR + DIR_SEP + "glerc";
			has_config = try_load_config(conf_name);
		} else {
			string conf_name2 = GLE_TOP_DIR + DIR_SEP + "glerc";
			has_config = try_load_config(conf_name2);
			if (has_config) {
				// only update name if configuration file found here
				conf_name = conf_name2;
			}
		}
	}
	check_correct_version(conf_name, has_top, has_config, coll);
	#if defined(__UNIX__) || defined (__OS2__)
		const char* home = getenv("HOME");
		if (home != NULL && home[0] != 0) {
			try_load_config(string(home) + DIR_SEP + ".glerc");
		}
	#endif
	// Set values for -v option
	init_installed_versions(cmdline, &coll);
	return has_config;
}

void do_find_deps(CmdLineObj& cmdline) {
	// Save config if find deps
	if (cmdline.hasOption(GLE_OPT_FINDDEPS)) {
		CmdLineArgString* arg = (CmdLineArgString*)cmdline.getOption(GLE_OPT_FINDDEPS)->getArg(0);
#ifdef __WIN32__
		if (arg->getValue().length() > 0) find_deps(arg->getValue(), &g_Config);
		else find_deps("C:\\Program Files", &g_Config);
		string conf_name = GLE_TOP_DIR + DIR_SEP + "glerc";
		try_save_config(conf_name, &g_Config);
#endif
		exit(0);
	}
}
