/***********************************************************************************
 * QGLE - A Graphical Interface to GLE                                             *
 * Copyright (C) 2006  A. S. Budden & J. Struyf                                    *
 *                                                                                 *
 * This program is free software; you can redistribute it and/or                   *
 * modify it under the terms of the GNU General Public License                     *
 * as published by the Free Software Foundation; either version 2                  *
 * of the License, or (at your option) any later version.                          *
 *                                                                                 *
 * This program is distributed in the hope that it will be useful,                 *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of                  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                   *
 * GNU General Public License for more details.                                    *
 *                                                                                 *
 * You should have received a copy of the GNU General Public License               *
 * along with this program; if not, write to the Free Software                     *
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA. *
 *                                                                                 *
 * Also add information on how to contact you by electronic and paper mail.        *
 ***********************************************************************************/

#include "arc.h"
#include "qgle_statics.h"

// The constructor for the line object
GLEArc::GLEArc(double resolution, int imageHeight, QObject *parent) :
	GLEDrawingObject(resolution, imageHeight, parent)
{
	directionReversed = false;
	// Make sure the line is updated if a point changes or 
	// the image size changes
	connect(this, SIGNAL(pointChanged()),
			this, SLOT(updateArc()));
	connect(this, SIGNAL(imageChanged()),
			this, SLOT(updateArc()));
}

// Update the painter path
void GLEArc::updateArc()
{
	QPointF p;
	QPair<QPointF,int> snap;
	// Only do this if both the centre and radius have been set
	if (isSet(CentrePoint) && isSet(StartPoint) && isSet(EndPoint))
	{
		// This is guaranteed to have been created in the constructor
		// of GLEDrawingObject
		delete(paintPath);
		// Create a new path starting at circleCentre
		paintPath = new QPainterPath(getQtPoint(CentrePoint));
		// Add the circle based on the qrect
		paintPath->arcTo(arcRect(), startAngleDeg(), QGLE::radiansToDegrees(endAngleRadConstrained()-startAngleRad()));

		// Update the GLE Code
		gleCode->clear();
		gleCode->append(QString("! %1")
				.arg(tr("Arc added using QGLE", "Comment added to source file")));
		gleCode->append(QString("asetpos %1")
				.arg(QGLE::GLEToStr(getGLEPoint(CentrePoint))));
		if (directionReversed)
			gleCode->append(QString("narc %1 %2 %3")
					.arg(QGLE::GLEToStr(getGLEDouble(Radius)))
					.arg(QGLE::GLEToStr(startAngleDeg()))
					.arg(QGLE::GLEToStr(endAngleDeg())));
		else
			gleCode->append(QString("arc %1 %2 %3")
					.arg(QGLE::GLEToStr(getGLEDouble(Radius)))
					.arg(QGLE::GLEToStr(startAngleDeg()))
					.arg(QGLE::GLEToStr(endAngleDeg())));
		gleCode->append(QString(""));



		// Now we add the osnap handles
		osnapHandles.clear();
		for(int i=0;i<360;i+=90)
		{
			if (isOnArc(QGLE::degreesToRadians(i)))
			{
				p.setX(getGLEDouble(Radius)*cos(QGLE::degreesToRadians(i)));
				p.setY(getGLEDouble(Radius)*sin(QGLE::degreesToRadians(i)));
				p += getGLEPoint(CentrePoint);
				snap.first = QGLE::absGLEToQt(p,dpi,pixmapHeight);
				snap.second = QGLE::QuadrantSnap;
				osnapHandles.append(snap);
			}
		}
		snap.first = getQtPoint(CentrePoint);
		snap.second = QGLE::CentreSnap;
		osnapHandles.append(snap);
		
		snap.first = getQtPoint(StartPoint);
		snap.second = QGLE::EndPointSnap;
		osnapHandles.append(snap);
		
		snap.first = getQtPoint(EndPoint);
		snap.second = QGLE::EndPointSnap;
		osnapHandles.append(snap);
		

	}
}

void GLEArc::addRelativeOSnaps(QPointF p)
{
	if (isSet(CentrePoint) && isSet(Radius) && isSet(StartPoint) && isSet(EndPoint))
	{
		qDebug() << "Adding osnaps relative to " << QGLE::absQtToGLE(p,dpi,pixmapHeight);
		QGLE::flushIO();

		relativeOSnaps.clear();

		// The first perpendicular osnap is defined as the nearest point (we'll check
		// whether it's on the circle later.
		QPointF np;
		distanceToPointOnCircle(p,&np);
		qDebug() << "Nearest point: " << QGLE::absQtToGLE(np,dpi,pixmapHeight);
		relativeOSnaps.append(QPair<QPointF,int>(np,QGLE::PerpendicularSnap));

		// The second perpendicular osnap is diametrically opposite the first one
		double angleToCentre = QGLE::angleBetweenTwoPoints(np,getQtPoint(CentrePoint));
		double radius = getQtDouble(Radius);
		double diameter = 2*radius;

		np = np + QPointF(diameter*cos(angleToCentre),diameter*sin(angleToCentre));
		relativeOSnaps.append(QPair<QPointF,int>(np,QGLE::PerpendicularSnap));

		double angle;

		if (!isInsideCircle(p))
		{
			// Now we need the tangential ones
			double distanceToCentre = QGLE::distance(p, getQtPoint(CentrePoint));
			double angleOffset = asin(radius/distanceToCentre);
			double distanceToTangent = sqrt(pow(distanceToCentre,2)-pow(radius,2));
			angleToCentre = QGLE::angleBetweenTwoPoints(p,getQtPoint(CentrePoint));

			angle = angleToCentre + angleOffset;
			np = p + QPointF(distanceToTangent*cos(angle),distanceToTangent*sin(angle));
			relativeOSnaps.append(QPair<QPointF,int>(np,QGLE::TangentSnap));

			angle = angleToCentre - angleOffset;
			np = p + QPointF(distanceToTangent*cos(angle),distanceToTangent*sin(angle));
			relativeOSnaps.append(QPair<QPointF,int>(np,QGLE::TangentSnap));
		}

		// The above is identical to the circle code, now check they're on the arc
		for(int i=relativeOSnaps.size()-1;i>=0;i--)
		{
			angle = QGLE::angleBetweenTwoPoints(getGLEPoint(CentrePoint),
					QGLE::absQtToGLE(relativeOSnaps[i].first,dpi,pixmapHeight));
			if (!isOnArc(angle))
				relativeOSnaps.removeAt(i);
		}
	}
	QGLE::flushIO();
}

void GLEArc::draw(QPainter *p)
{
	int sweep;
	if (selected)
	{
		// Draw the arc as a selected object
		QPen selPen;
		selPen = pen();
		selPen.setColor(Qt::red);
//		selPen.setStyle(Qt::DotLine);
		p->setPen(selPen);
	}
	else
	{
		// Draw as normal
		p->setPen(pen());
	}
	
	// If we don't have a start point AND an end point, give up now
	if (!(isSet(StartPoint) && isSet(EndPoint)))
		return;

	if (isSet(CentrePoint))
	{
		// If we have a centre point then we can draw an arc

		// Calculate the sweep
		sweep = (int) ((endAngleDegConstrained()-startAngleDeg())*16);
		if (directionReversed)
		{
			sweep = sweep-(360*16);
		}

		// Draw the arc
		p->drawArc(arcRect(), (int) (startAngleDeg()*16), sweep);
		//qDebug() << arcRect() << startAngleDeg()*16 << sweep << dpi << pixmapHeight;
	}
	else
	{
		// If we don't have a centre point, just draw a line
		p->drawLine(getQtPoint(StartPoint),getQtPoint(EndPoint));
	}
}

double GLEArc::distanceToPointOnCircle(QPointF p, QPointF *nearestPoint)
{
	QPointF c = getQtPoint(CentrePoint);
	if (nearestPoint)
	{
		double r = getQtDouble(Radius);
		double theta = QGLE::angleBetweenTwoPoints(c,p);
		nearestPoint->setX(r*cos(theta)+c.x());
		nearestPoint->setY(r*sin(theta)+c.y());
	}
	// Calculations in QT coordinates
	return(fabs(QGLE::distance(c,p)-getQtDouble(Radius)));
}

double GLEArc::distanceToPoint(QPointF p, QPointF *nearestPoint)
{
	// Calculations in GLE and Qt coordinates (to make the angles work!)
	double s, e, angle;
	QPointF cp, sp, ep;
	QPointF gleP = QGLE::absQtToGLE(p, dpi, pixmapHeight);
	QPointF np;

	// We need all the points, so if any are missing, return a very large number!
	if (!(isSet(CentrePoint) && isSet(Radius) && isSet(StartPoint) && isSet(EndPoint)))
		return(1e6);

	sp = getQtPoint(StartPoint);
	ep = getQtPoint(EndPoint);

	// Get the centre point and calculate the angle between the current point and it
	cp = getGLEPoint(CentrePoint);
	angle = QGLE::angleBetweenTwoPoints(cp,gleP);

	// If the point is on the arc (in terms of angle), the calculation is simple
	if (isOnArc(angle))
	{
		if (nearestPoint)
		{
			double r = getGLEDouble(Radius);
			np.setX(r*cos(angle)+cp.x());
			np.setY(r*sin(angle)+cp.y());
			*nearestPoint = QGLE::absGLEToQt(np, dpi, pixmapHeight);
		}

		cp = getQtPoint(CentrePoint);
		return(fabs(QGLE::distance(cp,p)-getQtDouble(Radius)));
	}

	
	// Otherwise, the distance will be the shortest distance to one of the end points
	s = QGLE::distance(p,sp);
	e = QGLE::distance(p,ep);

	if (s < e)
	{
		if (nearestPoint)
			*nearestPoint = sp;
		return(s);
	}
	else
	{
		if (nearestPoint)
			*nearestPoint = ep;
		return(e);
	}
}

bool GLEArc::isOnArc(double angle)
{
	// Decide whether an angle is on the drawn part of an arc
	double s, e;

	if (!(isSet(CentrePoint) && isSet(Radius) && isSet(StartPoint) && isSet(EndPoint)))
		return(false);


	// Choose the start and end angles according to the direction of drawing
	if (!directionReversed)
	{
		s = startAngleRad();
		e = endAngleRad();
	}
	else
	{
		s = endAngleRad();
		e = startAngleRad();
	}

	// Ensure that the end point is greater than the start point
	while (e < s)
		e += 2*M_PI;

	int i;

	// Decide whether the angle is in between the two points
	for (i=0;i<2;i++)
	{
		angle += i*2*M_PI;
		if ((angle >= s) && (angle <= e))
			return(true);
	}

	return(false);
	
}

// Set a point (start or end in the case of a line)
void GLEArc::setPoint(int pointChoice, QPointF p)
{
	QPointF s,e,m,c,mid1,mid2;
	double mg1,mg2,x,y;
	bool ok = false;
	switch(pointChoice)
	{

		// Set the radius based on a centre point
		case CentrePoint:
			s = getGLEPoint(StartPoint);
			x = QGLE::distance(p,s);
			if (x > 0.0)
				pointHash[Radius] = QPointF(x,0);
			pointHash[CentrePoint] = QGLE::absQtToGLE(p, dpi, pixmapHeight);
			break;

		case MidPoint:
			// Find the centre point and radius based on a random
			// point on the arc
			if (!(isSet(StartPoint) && isSet(EndPoint)))
				return;

			// Get the basic points in GLE coordinates
			s = getGLEPoint(StartPoint);
			e = getGLEPoint(EndPoint);
			m = QGLE::absQtToGLE(p, dpi,pixmapHeight);

			// Find the halfway points
			mid1 = QPointF((s.x()+m.x())/2.0,
					(s.y()+m.y())/2.0);
			mid2 = QPointF((e.x()+m.x())/2.0,
					(e.y()+m.y())/2.0);

			if ((m.y() == s.y()) && (m.x() != s.x()) && (m.y() != e.y()))
			{
				// start and mid point are on a horizontal line, but
				// points are not coincident or on the same line as
				// s - e
				//qDebug() << "1: " << s << m << e;

				mg2 = (e.x()-m.x())/(m.y()-e.y());
				x = mid1.x();
				y = mg2*x + mid2.y() - mg2*mid2.x();

				c = QPointF(x,y);

				pointHash[CentrePoint] = c;
				pointHash[Radius] = QPointF(QGLE::distance(c,m),0);

				if (!isOnArc(QGLE::angleBetweenTwoPoints(c,m)))
					directionReversed = !directionReversed;

				ok = true;
			
			}
			else if ((m.y() == e.y()) && (m.x() != e.x()) && (m.y() != s.y()))
			{
				//qDebug() << "2: " << s << m << e;
				// end and mid point are on a horizontal line, but
				// points are not coincident or on the same line as
				// s - e
				mg1 = (s.x()-m.x())/(m.y()-s.y());
				x = mid2.x();
				y = mg1*x + mid1.y() - mg1*mid1.x();

				c = QPointF(x,y);

				pointHash[CentrePoint] = c;
				pointHash[Radius] = QPointF(QGLE::distance(c,m),0);

				if (!isOnArc(QGLE::angleBetweenTwoPoints(c,m)))
					directionReversed = !directionReversed;

				ok = true;
			}
			else if ((m.y() != e.y()) && (m.y() != s.y()))
			{
				//qDebug() << "3";
				// The most common case: points are nicely spaced
				
				// Find the gradients of the normals to the intersecting lines
				mg1 = (s.x()-m.x())/(m.y()-s.y());
				mg2 = (e.x()-m.x())/(m.y()-e.y());

				if (!(mg1 == mg2))
				{
					x = (mid2.y() - mg2*mid2.x() - mid1.y() + mg1*mid1.x())/
						(mg1 - mg2);
					y = mg1*x + mid1.y() - mg1*mid1.x();

					c = QPointF(x,y);

					pointHash[CentrePoint] = c;
					pointHash[Radius] = QPointF(QGLE::distance(c,m),0);

					if (!isOnArc(QGLE::angleBetweenTwoPoints(c,m)))
						directionReversed = !directionReversed;

					ok = true;
				}
			}

			if (!ok)
			{
				pointHash.remove(CentrePoint);
				pointHash.remove(Radius);
			}

			

			// The simple cases:
		case StartPoint:
		case EndPoint:
			pointHash[pointChoice] = QGLE::absQtToGLE(p, dpi, pixmapHeight);
			break;
	}

	// Update the arc
	updateArc();
	QGLE::flushIO();
}

double GLEArc::startAngleDeg()
{
	// Return the angle between the centre and the start point
	// in degrees
	return(QGLE::radiansToDegrees(startAngleRad()));

}

double GLEArc::endAngleDeg()
{
	// Return the angle between the centre and the end point
	// in degrees
	return(QGLE::radiansToDegrees(endAngleRad()));

}

double GLEArc::endAngleDegConstrained()
{
	// Return the angle between the centre and the end point
	// in degrees and constrained to be greater than the
	// start point (by adding 2*pi)
	double e = endAngleDeg();
	double s = startAngleDeg();
	while (e < s)
		e += 360.0;
	return(e);
}

double GLEArc::startAngleRad()
{
	// Return the angle between the centre and the start point
	// in radians
	QPointF s = getGLEPoint(StartPoint);
	QPointF c = getGLEPoint(CentrePoint);
	return(QGLE::angleBetweenTwoPoints(c,s));
}

double GLEArc::endAngleRad()
{
	// Return the angle between the centre and the end point
	// in radians
	QPointF e = getGLEPoint(EndPoint);
	QPointF c = getGLEPoint(CentrePoint);
	return(QGLE::angleBetweenTwoPoints(c,e));
}

double GLEArc::endAngleRadConstrained()
{
	// Return the angle between the centre and the end point
	// in radians and constrained to be greater than the start
	// start point (by adding 2*pi)
	double e = endAngleRad();
	double s = startAngleRad();
	while (e < s)
		e += 2*M_PI;
	return(e);
}


QRectF GLEArc::arcRect()
{
	// Return the rectangle surrounding the circle upon which the arc is based 
	
	// Work in Qt coordinates
	QPointF cp = getQtPoint(CentrePoint);
	double r = getQtDouble(Radius);

	if ((isSet(Radius) && isSet(CentrePoint)))
		return(QRectF(cp.x()-r, cp.y() - r, 2*r, 2*r));
	else
		return(QRectF(0,0,0,0));

}


QList<QPointF> GLEArc::intersections(double qtm, double qtc, bool vertical)
{
	QPointF one, two;
	// For circles and arcs, we'll deal with points individually:
	if (vertical)
	{
		one.setX(qtm);
		two.setX(qtm);
		one.setY(0.0);
		two.setY(1.0);
	}
	else
	{
		one.setX(0.0);
		one.setY(qtc);
		two.setX(1.0);
		two.setY(qtm + qtc);
	}

	return(intersections(one,two));
}


QList<QPointF> GLEArc::intersections(QPointF qtp1, QPointF qtp2)
{
	// Based on http://tinyurl.com/qtgum
	QList<QPointF> pointArray;

	QPointF cp = getQtPoint(CentrePoint);
	double r = getQtDouble(Radius);
	double a,b,c;
	double bac;
	double u;
	double angle;
	QPointF p;

	a = pow((qtp2.x() - qtp1.x()),2) + pow((qtp2.y() - qtp1.y()),2);
	b = 2 * ( (qtp2.x() - qtp1.x())*(qtp1.x()-cp.x()) + (qtp2.y()-qtp1.y())*(qtp1.y()-cp.y()));
	c = pow(cp.x(),2)+pow(cp.y(),2)+pow(qtp1.x(),2)+pow(qtp1.y(),2)
		- 2*(cp.x()*qtp1.x()+cp.y()*qtp1.y()) - pow(r,2);

	bac = pow(b,2)-4*a*c;
	if (bac == 0.0)
	{
		u = - b / (2*a);
		p.setX(qtp1.x()+u*(qtp2.x()-qtp1.x()));
		p.setY(qtp1.y()+u*(qtp2.y()-qtp1.y()));

		angle = QGLE::angleBetweenTwoPoints(getGLEPoint(CentrePoint), 
				QGLE::absQtToGLE(p,dpi,pixmapHeight));
		if (isOnArc(angle))
			pointArray.append(p);
	}
	else if (bac > 0.0)
	{
		u = (-b + sqrt(bac))/(2*a);
		p.setX(qtp1.x() + u*(qtp2.x()-qtp1.x()));
		p.setY(qtp1.y() + u*(qtp2.y()-qtp1.y()));
		angle = QGLE::angleBetweenTwoPoints(getGLEPoint(CentrePoint), 
				QGLE::absQtToGLE(p,dpi,pixmapHeight));
		if (isOnArc(angle))
			pointArray.append(p);
		u = (-b - sqrt(bac))/(2*a);
		p.setX(qtp1.x() + u*(qtp2.x()-qtp1.x()));
		p.setY(qtp1.y() + u*(qtp2.y()-qtp1.y()));
		angle = QGLE::angleBetweenTwoPoints(getGLEPoint(CentrePoint), 
				QGLE::absQtToGLE(p,dpi,pixmapHeight));
		if (isOnArc(angle))
			pointArray.append(p);
	}

	return(pointArray);
}

QList<QPointF> GLEArc::intersections(QPointF qtp1, double angle)
{
	// This intersection code must determine the intersections in
	// a particular direction from a start point
	
	// First get a list based on an infinite line:
	QPointF qtp2 = qtp1 + QPointF(1.0*cos(angle),1.0*sin(angle));
	QList<QPointF> allIntersections = intersections(qtp1,qtp2);

	// Now go through the list and determine which are in the right
	// direction
	QList<QPointF> correctIntersections;
	QPointF pt;
	double ptAngle;

	foreach(pt, allIntersections)
	{
		ptAngle = QGLE::angleBetweenTwoPoints(qtp1, pt);
		if (QGLE::quadrant(ptAngle) == QGLE::quadrant(angle))
			correctIntersections.append(pt);
	}

	return(correctIntersections);

}

bool GLEArc::isInside(QPointF p)
{
	QPointF cp = getGLEPoint(CentrePoint);
	QPointF gp = QGLE::absQtToGLE(p,dpi,pixmapHeight);
	double angle;
	if (QGLE::distance(cp,gp) < getGLEDouble(Radius))
	{
		angle = QGLE::angleBetweenTwoPoints(cp,gp);
		if (isOnArc(angle))
			return(true);
	}

	return(false);
}

bool GLEArc::isInsideCircle(QPointF p)
{
	QPointF cp = getQtPoint(CentrePoint);
	if (QGLE::distance(cp,p) < getQtDouble(Radius))
		return(true);
	else
		return(false);
}
