/***********************************************************************************
 * QGLE - A Graphical Interface to GLE                                             *
 * Copyright (C) 2006  A. S. Budden & J. Struyf                                    *
 *                                                                                 *
 * This program is free software; you can redistribute it and/or                   *
 * modify it under the terms of the GNU General Public License                     *
 * as published by the Free Software Foundation; either version 2                  *
 * of the License, or (at your option) any later version.                          *
 *                                                                                 *
 * This program is distributed in the hope that it will be useful,                 *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of                  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                   *
 * GNU General Public License for more details.                                    *
 *                                                                                 *
 * You should have received a copy of the GNU General Public License               *
 * along with this program; if not, write to the Free Software                     *
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA. *
 *                                                                                 *
 * Also add information on how to contact you by electronic and paper mail.        *
 ***********************************************************************************/

#include <QtGui>
#include <QtDebug>

#include "gledrawing.h"
#include "drawingobject.h"
#include "qgle_definitions.h"
#include "qgle_statics.h"

// The constructor for the drawing area
GLEDrawingArea::GLEDrawingArea(QWidget *parent)
	: QWidget(parent)
{
	mainWindow = parent;

	// Initialise variables
	gridVisible = false;
	gridSnap = false;
	gridSpacing = QPointF(1.0,1.0);
	dpi = 0.0;
	isDrawing = false;
	hasAMove = false;
	snappedPosition = QPointF(0.0,0.0);
	lastPoint = QPointF(0.0,0.0);
	clearDirty();
	newObjects = false;
	objectSnap = false;
	orthoSnap = false;
	polarSnap = false;
	polarSnapFound = false;
	polarStartAngle = 0.0;
	polarAngleIncrement = 30.0;

	// We haven't selected an object yet
	objectSelected = false;
	amoveSelected = false;

	// Assume that keys aren't pressed at start up
	modifier[ShiftKey] = false;
	modifier[AltKey] = false;
	modifier[CtrlKey] = false;

	// Default to no tool
	currentTool = NoTool;

	// Grid uses QT units, but grid spacing is set in GLE units
	grid = new GLEGrid(this);
	grid->setGrid(gridSpacing);
	grid->setDPI(dpi);

	connect(this, SIGNAL(sizeChanged(QSize)),
			grid, SLOT(setArea(QSize)));
	connect(this, SIGNAL(dpiChanged(double)),
			grid, SLOT(setDPI(double)));

	// This ensures that Mouse Move events happen
	// even when the mouse buttons aren't pressed, which
	// is essential for updating the status bar and using
	// snap.
	setMouseTracking(true);

	// Make some handy messages to let the user know what to do
	statusMessages[NoTool][StartPoint] = tr("");
	statusMessages[PointerTool][StartPoint] = tr("Select an object, or hold shift to select multiple objects");
	statusMessages[LineTool][StartPoint] = tr("Select start point");
	statusMessages[CircleTool][StartPoint] = tr("Select centre point");
	statusMessages[ArcTool][StartPoint] = tr("Select start point");
	statusMessages[AMoveTool][StartPoint] = tr("Select final point for GLE script");

	statusMessages[LineTool][EndPoint] = tr("Select end point");
	statusMessages[ArcTool][EndPoint] = tr("Select end point");
	statusMessages[ArcTool][MidPoint] = tr("Select a point somewhere on the arc");
	statusMessages[CircleTool][CircumferencePoint] = tr("Select a point somewhere on the circle");
}

void GLEDrawingArea::setPolarSnapStartAngle(double newAngle)
{
	polarStartAngle = newAngle;
}

void GLEDrawingArea::setPolarSnapIncAngle(double newAngle)
{
	polarAngleIncrement = newAngle;
}

void GLEDrawingArea::modifierToggle(int mod, bool state)
{
	modifier[mod] = state;
}

// To allow us to notice whether changes have occurred
bool GLEDrawingArea::isDirty()
{
	return(dirty);
}

// To allow us to notice whether changes have occurred
void GLEDrawingArea::clearDirty()
{
	dirty = false;
	// The following sets the '*' on the title bar
	mainWindow->setWindowModified(false);
	qDebug() << "Dirty flag cleared";
	emit dirtyFlagChanged(dirty);
}

// To allow us to notice whether changes have occurred
void GLEDrawingArea::setDirty()
{
	dirty = true;
	mainWindow->setWindowModified(true);
	qDebug() << "Dirty flag set";
	emit dirtyFlagChanged(dirty);
}

void GLEDrawingArea::paintEvent(QPaintEvent *)
{
	// Create a QPainter to draw on the widget
	QPainter painter(this);
	// Fill the widget with white.
	painter.fillRect(rect(), Qt::white);

	QPointF nearestPoint;

	// If there is a pixmap available to draw,
	if (pixmap.isNull()) 
	{
		return;
	}
	// Always draw the pixmap first as it'll cover anything else
	painter.drawPixmap(QPointF(0,0),pixmap);

	if (isEditMode)
	{

		// Always draw the grid next, so that objects are over
		// the top of the grid.
		painter.save();
		if (gridVisible)
		{
			painter.setPen(grid->pen());

			QList<QPointF> gridPoints = grid->allPoints();

			for (int i=0;i<gridPoints.size();i++)
			{
				painter.drawPoint(gridPoints.at(i));
			}
			

		}
		if ((gridSnap || objectSnap || polarSnapFound) && (currentTool != PointerTool))
		{
			// Draw the cursor unless we're using the selection tool
			QPen cursorPen = grid->pen();
			cursorPen.setColor(Qt::black);
			cursorPen.setWidth(1);
			painter.setPen(cursorPen);

			int crossLength = 4;
			// Draw a cross at the nearest grid point
			QGLE::drawCross(&painter, snappedPosition, crossLength, cursorPen);
		}
		painter.restore();

		QPen osnapPen;
		osnapPen.setColor(Qt::darkYellow);
		osnapPen.setWidth(1);

		if (polarSnap && !gridSnap && !orthoSnap)
		{
			SnapLine *snap;
			foreach(snap,snapLineList)
			{
				snap->draw(&painter);
			}
			QPointF snapPoint;
			foreach(snapPoint, snapLinePoints)
			{
				QGLE::drawBox(&painter, snapPoint, OSNAP_BOX_SIZE, osnapPen);
			}
		}

		// If we have any objects, draw them.  At this level, we have
		// no information as to whether the object is a line, circle,
		// arc or whatever: we just use the common functionality.
		if (thereAreObjects())
		{
			for(int i=0;i<objectList.size();i++)
			{
				objectList[i]->draw(&painter);
				if (objectSnap && (currentTool != PointerTool))
				{
					if ((orthoSnap || polarSnapFound) && (osnapPoints.size() > 0))
					{
						QPointF p;
						foreach(p,osnapPoints)
						{
							QGLE::drawBox(&painter, p, OSNAP_BOX_SIZE, osnapPen);
						}
					}
					else
					{
						if ((i != currentIndex) || !isDrawing)
						{
							if (objectList[i]->distanceToPoint(cursorPosition, &nearestPoint) < MAX_OSNAP_DRAW_DISTANCE)
							{
								objectList[i]->drawOSnaps(&painter);
								QGLE::drawBox(&painter, nearestPoint, OSNAP_BOX_SIZE, osnapPen);
							}
							else if (objectList[i]->isInside(cursorPosition))
								objectList[i]->drawOSnaps(&painter);
						}
					}
				}
			}
		}

		// Draw a cross to show where the final point is
		if (hasAMove)
		{
			QPen amovePen;
			if (amoveSelected)
				amovePen.setColor(Qt::red);
			else
				amovePen.setColor(Qt::black);
			amovePen.setWidth(1);
			painter.setPen(amovePen);
			QPointF qtAmove = QGLE::absGLEToQt(amove, dpi, pixmap.height());
			painter.drawLine(QLineF(qtAmove.x()-AMOVE_LENGTH,
						qtAmove.y()-AMOVE_LENGTH,
						qtAmove.x()+AMOVE_LENGTH,
						qtAmove.y()+AMOVE_LENGTH));
			painter.drawLine(QLineF(qtAmove.x()-AMOVE_LENGTH,
						qtAmove.y()+AMOVE_LENGTH,
						qtAmove.x()+AMOVE_LENGTH,
						qtAmove.y()-AMOVE_LENGTH));
		}
	}

}

// Are there any objects to draw?
bool GLEDrawingArea::thereAreObjects()
{
	return(!objectList.isEmpty());
}


// Get rid of any objects that have been drawn
void GLEDrawingArea::clearObjects()
{
	objectSelected = false;
	hasAMove = false;
	GLEDrawingObject *obj;
	foreach (obj, objectList)
	{
		delete obj;
	}
	objectList.clear();
	qDebug() << "Object list cleared";
}

// Return a list of the GLE code necessary to draw the
// added objects
QStringList GLEDrawingArea::gleObjects()
{
	QStringList gleCode;

	// The objects
	if (!objectList.isEmpty())
	{
		GLEDrawingObject *object;
		foreach(object, objectList)
			gleCode << object->GLECode();
	}
	
	// The final point
	if (hasAMove)
	{
		QString amoveCmd = "amove ";
		amoveCmd += QGLE::GLEToStr(amove);
		gleCode << amoveCmd;
	}
	return(gleCode);
}

// Update the resolution and notify children
void GLEDrawingArea::setDPI(double new_dpi)
{
	if ( dpi != new_dpi)
	{
		dpi = new_dpi;
		// Emit a signal to all daughters
		emit dpiChanged(dpi);
	}
}

// Update the grid spacing
void GLEDrawingArea::setGrid(QPointF newGrid)
{
	if ( gridSpacing != newGrid)
	{
		gridSpacing = newGrid;
		grid->setGrid(gridSpacing);
		update();
	}
}

// This slot receives an image from the server thread
// and updates the display accordingly.
void GLEDrawingArea::updateDisplay(QImage image)
{
	// Check for changes in pixmap height
	int oldheight = pixmap.height();
	int oldwidth = pixmap.width();

	pixmap = QPixmap::fromImage(image);

	// Notify children if the height has changed:
	// used for coordinate transforms
	if ((oldheight != pixmap.height()) || (oldwidth != pixmap.width()))
		emit sizeChanged(pixmap.size());

	// Resize the drawing area to the size of the pixmap
	resize(pixmap.width(),pixmap.height());

	// Update the display
	update();
}

// Toggle the visibility of the grid
void GLEDrawingArea::gridToggle(bool state)
{
	gridVisible = state;
	update();
}

void GLEDrawingArea::osnapToggle(bool state)
{
	objectSnap = state;
	update();
}

void GLEDrawingArea::polarSnapToggle(bool state)
{
	polarSnap = state;
	update();
}

void GLEDrawingArea::orthoSnapToggle(bool state)
{
	orthoSnap = state;
	// Will probably have to edit the current line here too!
	update();
}

double GLEDrawingArea::getDPI()
{
	return(dpi);
}

// Toggle grid snap
void GLEDrawingArea::gridSnapToggle(bool state)
{
	gridSnap = state;
	update();
}

void GLEDrawingArea::createPolarSnaps(QPointF qtPoint)
{
	SnapLine *snap;
	qDebug() << "Creating snaps from point " << qtPoint;
	for(double i = polarStartAngle; i < 360.0 ; i += polarAngleIncrement)
	{
		qDebug() << "Creating snap line at angle " << i;
		snap = new SnapLine(dpi, pixmap.height(), this, pixmap.size());
		connect(this, SIGNAL(sizeChanged(QSize)),
				snap, SLOT(setPageSize(QSize)));
		snap->setPoint(SnapLine::StartPoint, qtPoint);
		snap->setPoint(SnapLine::Angle, QPointF(QGLE::degreesToRadians(i),0.0));
		snapLineList.append(snap);
	}
}

void GLEDrawingArea::clearPolarSnaps()
{
	SnapLine *snap;
	foreach(snap, snapLineList)
	{
		delete snap;
	}
	snapLineList.clear();
	snapLinePoints.clear();
	polarSnapFound = false;
}

// Event called when a mouse button is pressed;
void GLEDrawingArea::mousePressEvent(QMouseEvent *event)
{
	if (currentTool == NoTool)
		return;
	QPointF qtPoint;
	if (event->button() == Qt::LeftButton)
	{
		clearPolarSnaps();
		switch (currentTool)
		{
			case LineTool:
				// Click to start; click to end.
				// The end point is continuously updated
				// in the mouseMoveEvent function
				if (isDrawing == true)
				{
					// Drawing finished
					isDrawing = false;
					GLEDrawingObject *obj;
					foreach(obj,objectList)
					{
						obj->clearRelativeOSnaps();
					}
					changeBasePoint(QPointF(0.0,0.0));
					emit updateStatusBar(statusMessages[LineTool][StartPoint]);
				}
				else
				{
					// Create a line
					GLELine *currentLine = new GLELine(dpi, pixmap.height(), this);
					// Add it to the array of lines
					objectList.append(currentLine);
					// Get the index
					currentIndex = objectList.size() - 1;

					// Make sure the line is updated when the image size changes
					connect(this, SIGNAL(dpiChanged(double)),
							objectList[currentIndex], SLOT(setDPI(double)));
					connect(this, SIGNAL(sizeChanged(QSize)),
							objectList[currentIndex], SLOT(setPixmapSize(QSize)));


					// Register the start point
					qtPoint = snappedPoint(event->pos());

					objectList[currentIndex]->setPoint(GLELine::StartPoint,
							qtPoint);
					objectList[currentIndex]->setPoint(GLELine::EndPoint,
							qtPoint);
					// Start drawing
					isDrawing = true;
					// Create snap lines from this point
					if(polarSnap)
						createPolarSnaps(qtPoint);

					changeBasePoint(QGLE::absQtToGLE(qtPoint, dpi, pixmap.height()));
					emit updateStatusBar(statusMessages[LineTool][EndPoint]);
			
				}

				setDirty();
				newObjects = true;
				break;

			case ArcTool:
				// Click for start, click for end, click for mid point
				if (isDrawing == true)
				{
					if (objectList[currentIndex]->isSet(GLEArc::CentrePoint))
					{
						// Then we're finished, so stop drawing
						isDrawing = false;
						GLEDrawingObject *obj;
						foreach(obj,objectList)
						{
							obj->clearRelativeOSnaps();
						}
						changeBasePoint(QPointF(0,0));
						emit updateStatusBar(statusMessages[ArcTool][StartPoint]);
					}
					else
					{
						qtPoint = snappedPoint(event->pos());

						objectList[currentIndex]->setPoint(GLEArc::MidPoint,
								qtPoint);
						if(polarSnap)
							createPolarSnaps(qtPoint);

						changeBasePoint(QGLE::absQtToGLE(qtPoint, dpi, pixmap.height()));
						emit updateStatusBar(statusMessages[ArcTool][MidPoint]);
						// Create snap lines from this point
					}
					
				}
				else
				{
					// Create an arc
					GLEArc *currentArc = new GLEArc(dpi, pixmap.height(), this);
					// Add it to the array of objects
					objectList.append(currentArc);
					// Get the index
					currentIndex = objectList.size() - 1;

					// Make sure the line is updated when the image size changes
					connect(this, SIGNAL(dpiChanged(double)),
							objectList[currentIndex], SLOT(setDPI(double)));
					connect(this, SIGNAL(sizeChanged(QSize)),
							objectList[currentIndex], SLOT(setPixmapSize(QSize)));

					// Register the start point
					qtPoint = snappedPoint(event->pos());
					objectList[currentIndex]->setPoint(GLEArc::StartPoint,
							qtPoint);
					objectList[currentIndex]->setPoint(GLEArc::EndPoint,
							qtPoint);
					// Start drawing
					isDrawing = true;
					// Create snap lines from this point
					if (polarSnap)
						createPolarSnaps(qtPoint);

					changeBasePoint(QGLE::absQtToGLE(qtPoint, dpi, pixmap.height()));
					emit updateStatusBar(statusMessages[ArcTool][EndPoint]);
			
				}

				setDirty();
				newObjects = true;
				break;


			case CircleTool:
				// Click for centre, click for point on circumference
				if (isDrawing == true)
				{
					// Drawing finished
					isDrawing = false;
					GLEDrawingObject *obj;
					foreach(obj,objectList)
					{
						obj->clearRelativeOSnaps();
					}
					changeBasePoint(QPointF(0,0));
					emit updateStatusBar(statusMessages[CircleTool][StartPoint]);
				}
				else
				{
					// Create a line
					GLECircle *currentCircle = new GLECircle(dpi, pixmap.height(), this);
					// Add it to the array of objects
					objectList.append(currentCircle);
					// Get the index
					currentIndex = objectList.size() - 1;

					// Make sure the line is updated when the image size changes
					connect(this, SIGNAL(dpiChanged(double)),
							objectList[currentIndex], SLOT(setDPI(double)));
					connect(this, SIGNAL(sizeChanged(QSize)),
							objectList[currentIndex], SLOT(setPixmapSize(QSize)));

					// Register the start point
					qtPoint = snappedPoint(event->pos());
					objectList[currentIndex]->setPoint(GLECircle::CentrePoint,
							qtPoint);
					// Start drawing
					isDrawing = true;
					// Create snap lines from this point
					if (polarSnap)
						createPolarSnaps(qtPoint);

					changeBasePoint(QGLE::absQtToGLE(qtPoint, dpi, pixmap.height()));
					emit updateStatusBar(statusMessages[CircleTool][CircumferencePoint]);

				}

				setDirty();
				newObjects = true;
				break;

			case AMoveTool:
				// Add an 'amove' at the end of the file
				hasAMove = true;

				snappedPosition = snappedPoint(event->pos());

				amove = QGLE::absQtToGLE(snappedPosition, dpi, pixmap.height());

				setDirty();
				newObjects = true;
				emit updateStatusBar(statusMessages[AMoveTool][StartPoint]);

				break;

			case PointerTool:

				// Select the nearest object (unless it's a long way away)
				double shortest_distance = 1e6;
				int shortest_index = -1;
				double distance;

				// Find the nearest object and the distance to it
				for(int i=0;i<objectList.size();i++)
				{
					distance = objectList[i]->distanceToPoint(event->pos());
					//qDebug() << "Distance to line " << i << ": " << distance;
					if (distance < shortest_distance)
					{
						shortest_distance = distance;
						shortest_index = i;
					}
				}
				// Now compare with the distance to the amove if there is one
				if (hasAMove)
				{
					distance = QGLE::distance(event->pos(),
							QGLE::absGLEToQt(amove,dpi,pixmap.height()));
					if (distance < shortest_distance)
					{
						shortest_distance = distance;
						shortest_index = -1;
					}
				}

				// If it's less than the predefined maximum, select that object
				if (shortest_distance < MAX_SELECT_DISTANCE)
				{
					qDebug() << "Closest Index: " << shortest_index;

//					if (objectSelected == true)
//						selectedObject->setSelected(false);

					if (shortest_index > -1)
					{
						//amoveSelected = false;
						if (!modifier[ShiftKey])
							unselectObjects();

						objectSelected = true;
						objectList[shortest_index]->setSelected(true);
					}
					else
					{
						if (!modifier[ShiftKey])
							unselectObjects();

						amoveSelected = true;
					}

					QGLE::flushIO();
				}
				else
				{
					// If it's more than the predefined maximum, unselect the objects
					if (!modifier[ShiftKey])
						unselectObjects();
				}

				emit updateStatusBar(statusMessages[PointerTool][StartPoint]);

				// Redraw the screen
		}
	}
	update();
}

void GLEDrawingArea::unselectObjects()
{
	amoveSelected = false;
	if (objectSelected == true)
	{
		GLEDrawingObject *obj;
		foreach(obj, objectList)
		{
			if (obj->isSelected())
				obj->setSelected(false);
		}
	}
	objectSelected = false;
}


// Clear the New Objects flag
void GLEDrawingArea::clearNewObjectsFlag()
{
	newObjects = false;
	qDebug() << "NEW Objects flag cleared";
}

void GLEDrawingArea::setNewObjectsFlag()
{
	newObjects = true;
	qDebug() << "NEW Objects flag set";
}

// Identify whether there are new objects
bool GLEDrawingArea::thereAreNewObjects()
{
	return(newObjects);
}

void GLEDrawingArea::updateSnapLines(QPointF qtPoint)
{
	QPointF nearestPoint;
	snapLinePoints.clear();
	for(int i=snapLineList.size()-1;i>=0;i--)
	{
		// Are we too far away to keep it?
		if (snapLineList[i]->distanceToPoint(qtPoint, &nearestPoint) > MAX_SNAP_LINE_DISTANCE)
		{
			if (QGLE::distance(qtPoint, 
						snapLineList[i]->getQtPoint(SnapLine::StartPoint)) > 3*MAX_SNAP_LINE_DISTANCE)
				snapLineList[i]->deactivate();
			else
			{
				delete snapLineList[i];
				snapLineList.removeAt(i);
			}
		}
		else
		{
			snapLineList[i]->activate();
			snapLinePoints.append(nearestPoint);

		}
	}

}

// Event called when the mouse moves over the drawing area
void GLEDrawingArea::mouseMoveEvent(QMouseEvent *event)
{
	// Get the mouse position in GLE coordinates
	QPointF glePosition;

	cursorPosition = event->pos();

	if (isDrawing)
		updateSnapLines(cursorPosition);

	snappedPosition = snappedPoint(cursorPosition);

	glePosition = QGLE::absQtToGLE(snappedPosition, dpi, pixmap.height());
	
	// Notify the world (currently just used to update the status bar)
	emit mouseMoved(glePosition);

	if (currentTool == NoTool)
		return;

	if(isDrawing)
	{
		switch (currentTool)
		{
			case LineTool:
				// If we're drawing, set the end point of the line to the 
				// current mouse position so that the line is drawn in the 
				// paintEvent.
				objectList[currentIndex]->setPoint(GLELine::EndPoint, snappedPosition);
				update();
				break;

			case CircleTool:
				// Set a point on the circumference of the circle
				objectList[currentIndex]->setPoint(GLECircle::CircumferencePoint, snappedPosition);
				update();
				break;
				
			case ArcTool:
				// Either set the end point of the line or the midpoint
				if (objectList[currentIndex]->isSet(GLEArc::MidPoint))
					objectList[currentIndex]->setPoint(GLEArc::MidPoint, snappedPosition);
				else
				{
					objectList[currentIndex]->setPoint(GLEArc::EndPoint, snappedPosition);
				}
				update();
				break;
		}

	}
	else 
	{
		if (gridSnap || objectSnap)
			update();
		emit updateStatusBar(statusMessages[currentTool][StartPoint]);
	}
}

void GLEDrawingArea::changeBasePoint(QPointF p)
{
	lastPoint = QGLE::absGLEToQt(p,dpi,pixmap.height());
	emit basePointChanged(p);

	if (isDrawing)
	{
		for(int i=0;i<objectList.size();i++)
		{
			if (i != currentIndex)
				objectList[i]->addRelativeOSnaps(lastPoint);
		}
	}
	else
	{
		GLEDrawingObject *obj;
		foreach(obj,objectList)
		{
			obj->clearRelativeOSnaps();
		}
	}
		
}

// Event called when the mouse button is released.
void GLEDrawingArea::mouseReleaseEvent(QMouseEvent *)
{
	QStringList code;
	switch (currentTool)
	{
		case CircleTool:
		case ArcTool:
		case LineTool:
			// This is for debugging: output the GLE code on the terminal
			code = objectList[currentIndex]->GLECode();
			for(int i=0;i<code.size();i++)
				qDebug(code.at(i).toLatin1().constData());
			break;

	}
}

// Change the tool
void GLEDrawingArea::setTool(int newTool)
{
	currentTool = newTool;
	
	// If we're not using a pointer tool, nothing should be
	// selected
	if (currentTool != PointerTool)
	{
		unselectObjects();
		update();
	}

	emit updateStatusBar(statusMessages[currentTool][StartPoint]);
}

// Are we in edit mode?
void GLEDrawingArea::setEditMode(bool state)
{
	isEditMode = state;
}

// Handle key presses
void GLEDrawingArea::hitKey(int key, int modifiers)
{
	qDebug() << "Key event: " << key;
	if (modifiers != Qt::NoModifier)
		return;

	switch(key)
	{
		// If we get an escape key, stop drawing
		case Qt::Key_Escape:
			if (isDrawing == true)
			{
				isDrawing = false;
				delete objectList[currentIndex];
				objectList.removeAt(currentIndex);
				changeBasePoint(QPointF(0.0,0.0));
				clearPolarSnaps();
				update();
			}
			break;

		// If we get a delete key and something is selected,
		// delete it.
		case Qt::Key_Delete:
			if (objectSelected)
			{
				for(int i=objectList.size()-1;i>=0;i--)			
				{
					if (objectList[i]->isSelected())
					{
						delete objectList[i];
						objectList.removeAt(i);
					}
				}
				objectSelected = false;
				newObjects = true;
				setDirty();
			}
			if (amoveSelected)
			{
				hasAMove = false;
				setDirty();
			}
			update();
			break;

	}
}

QPointF GLEDrawingArea::snappedPoint(QPointF p)
{
	// This function is intended to find the appropriate snapping 
	// point (if there is one).
	
	QPointF orthoPoint;
	bool horiz; // Horizontal ortho?
	polarSnapFound = false;

	osnapPoints.clear();
	if (gridSnap)
		snappedPosition = grid->nearestPoint(p);
	else
		snappedPosition = p;

	// Find the point where the orthoSnap meets the grid
	// ORTHO SNAP overrides everything else!
	if (orthoSnap && isDrawing)
	{
		double upDist = QGLE::distance(p,lastPoint)*sin(QGLE::angleBetweenTwoPoints(p,lastPoint));
		double acrossDist = QGLE::distance(p,lastPoint)*cos(QGLE::angleBetweenTwoPoints(p,lastPoint));
		if (fabs(upDist) > fabs(acrossDist))
		{
			horiz = false;
			orthoPoint = lastPoint - QPointF(0.0,upDist);
		}
		else
		{
			horiz = true;
			orthoPoint = lastPoint - QPointF(acrossDist,0.0);
		}

		if (gridSnap)
		{
			if (horiz)
				orthoPoint.setX(snappedPosition.x());
			else
				orthoPoint.setY(snappedPosition.y());
		}
		else if (objectSnap)
		{
			// Where does the ortho line cross any objects - save regardless
			// of whether we're close enough

			if (thereAreObjects())
			{
				for(int i=0;i<objectList.size();i++)
				{
					if ((i != currentIndex) || (!isDrawing))
					{
						if (horiz)
						{
							if (p.x() > lastPoint.x())
								osnapPoints += objectList[i]->intersections(lastPoint,
										QGLE::degreesToRadians(0.0));
							else
								osnapPoints += objectList[i]->intersections(lastPoint,
										QGLE::degreesToRadians(180.0));
						}
						else
						{
							if (p.y() > lastPoint.y())
								osnapPoints += objectList[i]->intersections(lastPoint,
										QGLE::degreesToRadians(90.0));
							else
								osnapPoints += objectList[i]->intersections(lastPoint,
										QGLE::degreesToRadians(270.0));
						}
					}
				}

				QPointF osnapPoint;
				QPointF closestPoint;
				double dist;
				double shortest_distance = 1e6;
				foreach(osnapPoint, osnapPoints)
				{
					dist = QGLE::distance(osnapPoint, orthoPoint);
					if (dist < shortest_distance)
					{
						shortest_distance = dist;
						closestPoint = osnapPoint;
					}
				}
				if (shortest_distance < MAX_OSNAP_DISTANCE)
					return(closestPoint);
			}

		}

		return(orthoPoint);
	}
	else
	{
		// Either orthosnap is off or we're not yet drawing, grid comes first!
		if (gridSnap)
			return(snappedPosition);

		// Now we'll try polar snap
		if (polarSnap)
		{
			SnapLine *snap;
			if (objectSnap)
			{
				if (thereAreObjects())
				{
					for(int i=0;i<objectList.size();i++)
					{
						if ((i != currentIndex) || (!isDrawing))
						{
							foreach(snap,snapLineList)
							{
								if (snap->isActive())
								{
									osnapPoints += objectList[i]->intersections(
											snap->getQtPoint(SnapLine::StartPoint),
											-snap->getGLEDouble(SnapLine::Angle));
								}
							}
						}
					}

					QPointF osnapPoint;
					QPointF closestPoint;
					double dist;
					double shortest_distance = 1e6;
					foreach(osnapPoint, osnapPoints)
					{
						dist = QGLE::distance(osnapPoint, p);
						if (dist < shortest_distance)
						{
							shortest_distance = dist;
							closestPoint = osnapPoint;
						}
					}
					if (shortest_distance < MAX_OSNAP_DISTANCE)
					{
						polarSnapFound = true;
						return(closestPoint);
					}
				}
			}
			// One can now assume that we're not close to the osnapPoints
			// Therefore, see how close we are to the snapLines
			double dist;
			double shortest_distance = 1e6;
			QPointF closestPoint;
			QPointF pt;
			foreach(snap, snapLineList)
			{
				dist = snap->distanceToPoint(p,&pt);
				if (dist < shortest_distance)
				{
					shortest_distance = dist;
					closestPoint = pt;
				}
			}
			if (shortest_distance < MAX_OSNAP_DISTANCE)
			{
				polarSnapFound = true;
				return(closestPoint);
			}

		}

		if (objectSnap)
		{
			// If the grid snap is off, we move onto OSNAP.  This is where it gets interesting!
			// Let the paintEvent cope with drawing the handles, just work out which
			// is closest
			double dist;
			double shortest_distance = 1e6;
			QPointF closestPoint;
			QPointF point;
			if (thereAreObjects())
			{
				for(int i=0;i<objectList.size();i++)
				{
					if ((i != currentIndex) || (!isDrawing))
					{
						dist = objectList[i]->nearestOSnap(p,&point);
						if (dist < shortest_distance)
						{
							shortest_distance = dist;
							closestPoint = point;
						}
					}
				}

				if (shortest_distance < MAX_OSNAP_DISTANCE)
					return(closestPoint);

				// If the standard handle tests fail, try the nearest point on the object
				for(int i=0;i<objectList.size();i++)
				{
					if ((i != currentIndex) || (!isDrawing))
					{
						dist = objectList[i]->distanceToPoint(p,&point);
						if (dist < shortest_distance)
						{
							shortest_distance = dist;
							closestPoint = point;
						}
					}
				}

				if (shortest_distance < MAX_OSNAP_DISTANCE)
					return(closestPoint);

			}
		}
					

	}

	// If nothing else has come up with a better option, return the point unchanged
	return(p);
}

