/***********************************************************************************
 * QGLE - A Graphical Interface to GLE                                             *
 * Copyright (C) 2006  A. S. Budden & J. Struyf                                    *
 *                                                                                 *
 * This program is free software; you can redistribute it and/or                   *
 * modify it under the terms of the GNU General Public License                     *
 * as published by the Free Software Foundation; either version 2                  *
 * of the License, or (at your option) any later version.                          *
 *                                                                                 *
 * This program is distributed in the hope that it will be useful,                 *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of                  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                   *
 * GNU General Public License for more details.                                    *
 *                                                                                 *
 * You should have received a copy of the GNU General Public License               *
 * along with this program; if not, write to the Free Software                     *
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA. *
 *                                                                                 *
 * Also add information on how to contact you by electronic and paper mail.        *
 ***********************************************************************************/

#ifndef __GLEDRAWING_H
#define __GLEDRAWING_H

#include <QtGui>
#include "line.h"
#include "circle.h"
#include "arc.h"
#include "snapline.h"
#include "grid.h"

//! Class implementing the drawing area for displaying GLE drawings
/*!
 * This class controls the paintable area of the window.  For the purposes of
 * the initial previewer, it is largely unnecessary, as all that is required
 * is for the picture to be drawn.  However, for future versions, there will
 * be additional usage, including user interaction and this will be cleaner
 * if implemented in a separate class.
 */
class GLEDrawingArea : public QWidget
{
	Q_OBJECT
	
public:
	//! Constructor
	GLEDrawingArea(QWidget *parent);

	//! Enumerator describing the drawing tools
	enum DrawingTools
	{
		NoTool,
		PointerTool,
		LineTool,
		CircleTool,
		ArcTool,
		AMoveTool
	};

	//! Enumerator used to list the modifiers that we use
	enum
	{
		ShiftKey,
		AltKey,
		CtrlKey
	};

	//! Return the current resolution
	double getDPI();

	//! Get the extra GLE code for all user drawn objects
	QStringList gleObjects();

	//! Has the image been changed
	bool isDirty();
	//! We've saved, so clear the dirty flag
	void clearDirty();
	//! Set the dirty flag
	void setDirty();
	//! Objects have been rendered
	void clearNewObjectsFlag();
	void setNewObjectsFlag();
	//! Are there new objects since the last render?
	bool thereAreNewObjects();
	//! Are there any objects at all?
	bool thereAreObjects();

	//! We've opened a new file, so clear the objects
	void clearObjects();

	//! Handle key events
	void hitKey(int key, int modifiers);

protected:
	//! Automatically called when repainting is needed
	void paintEvent(QPaintEvent *event);
	//! Automatically called when a mouse button is pressed
	void mousePressEvent(QMouseEvent *event);
	//! Automatically called when the mouse is moved
	void mouseMoveEvent(QMouseEvent *event);
	//! Automatically called when a mouse button is released
	void mouseReleaseEvent(QMouseEvent *event);
 
public slots:
	//! Slot used to update the display when GS is finished
	void updateDisplay(QImage image);
	//! Slot used to toggle the visibility of the grid
	void gridToggle(bool state);
	//! Slot used to toggle the grid snap
	void gridSnapToggle(bool state);
	//! Slot used to update the resolution of the drawing
	void setDPI(double new_dpi);
	//! Slot used to change the grid X spacing
	void setGrid(QPointF newGrid);
	//! Slot used to select a new drawing tool
	void setTool(int newTool);
	//! Slot used to select a mode
	void setEditMode(bool state);
	//! Slot used to toggle orthosnap
	void orthoSnapToggle(bool state);
	//! Slot used to toggle osnap
	void osnapToggle(bool state);
	//! Slot used to toggle polar snap
	void polarSnapToggle(bool state);
	//! Slot used to handle keyboard modifiers
	void modifierToggle(int mod, bool state);
	//! Slot used to update the polar start angle
	void setPolarSnapStartAngle(double newAngle);
	//! Slot used to update the polar increment angle
	void setPolarSnapIncAngle(double newAngle);

signals:
	//! Notify children of change of resolution
	void dpiChanged(double new_dpi);
	//! Notify children of change of image size
	void sizeChanged(QSize new_size);
	//! Notify parent's status bar of mouse move events
	void mouseMoved(QPointF gle);
	//! Notify parent of base point
	void basePointChanged(QPointF qt);
	//! Update status bar as things are done
	void updateStatusBar(QString msg);
	//! Notify of changes to the dirty flag
	void dirtyFlagChanged(bool state);

private:
	//! The rendered pixmap
	QPixmap pixmap;
	//! The visibility of the grid
	bool gridVisible;
	//! Whether we're snapping to the grid
	bool gridSnap;
	//! The grid spacing
	QPointF gridSpacing;
	//! The resolution
	double dpi;

	//! Hash holding modifier states
	QHash<int,bool> modifier;

	//! The Grid
	GLEGrid *grid;

	//! Choose a point based on all the snaps
	QPointF snappedPoint(QPointF p);

	//! Change the base point
	void changeBasePoint(QPointF p);

	//! Unselect all objects
	void unselectObjects();

	QPointF lastPoint;
	QList<QPointF> osnapPoints;

	//! Are any objects selected?
	bool objectSelected;
	//! Is the amove selected
	bool amoveSelected;
	
	//! The current cursor position
	QPointF cursorPosition;
	
	//! The nearest grid point (equal to the above if no grid snap)
	QPointF snappedPosition;

	//! Is orthosnap on
	bool orthoSnap;

	//! Is polar snap on?
	bool polarSnap;

	//! The index of the current line in the line list
	int currentIndex;

	bool polarSnapFound;

	//! The current tool being used
	int currentTool;

	//! The list of lines
	QList<GLEDrawingObject *> objectList;

	//! The list of snap lines
	QList<SnapLine *> snapLineList;
	//! The list of points for the snap line list
	QList<QPointF> snapLinePoints;

	void createPolarSnaps(QPointF qtPoint);
	void clearPolarSnaps();
	void updateSnapLines(QPointF qtPoint);

	//! A list of status bar messages
	QHash<int, QHash<int, QString> > statusMessages;

	//! The amove at the end of the script
	QPointF amove;
	//! Is it present?
	bool hasAMove;

	//! Are we drawing?
	bool isDrawing;

	//! Are we in an editing mode
	bool isEditMode;

	//! Are we in object snap mode
	bool objectSnap;

	//! Has the drawing been changed
	bool dirty;
	//! Has it been changed since the last render
	bool newObjects;

	//! A record of the parent
	QWidget *mainWindow;

	//! Enumerator used to describe what we're drawing (for status bar messages)
	enum
	{
		StartPoint,
		MidPoint,
		EndPoint,
		CircumferencePoint
	};

	double polarStartAngle;
	double polarAngleIncrement;

};


#endif
