/***********************************************************************************
 * QGLE - A Graphical Interface to GLE                                             *
 * Copyright (C) 2006  A. S. Budden & J. Struyf                                    *
 *                                                                                 *
 * This program is free software; you can redistribute it and/or                   *
 * modify it under the terms of the GNU General Public License                     *
 * as published by the Free Software Foundation; either version 2                  *
 * of the License, or (at your option) any later version.                          *
 *                                                                                 *
 * This program is distributed in the hope that it will be useful,                 *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of                  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                   *
 * GNU General Public License for more details.                                    *
 *                                                                                 *
 * You should have received a copy of the GNU General Public License               *
 * along with this program; if not, write to the Free Software                     *
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA. *
 *                                                                                 *
 * Also add information on how to contact you by electronic and paper mail.        *
 ***********************************************************************************/

#include "grid.h"
#include "qgle_definitions.h"
#include <math.h>

// The constructor, initialise variables and clear the grid
GLEGrid::GLEGrid(QObject *parent) : QObject(parent)
{
	gridSpacing = QPointF(0.0,0.0);
	dpi = 0;

	paintPen.setStyle(Qt::SolidLine);
	paintPen.setCapStyle(Qt::RoundCap);
	paintPen.setColor(Qt::black);
	//paintPen.setWidth(0);
	paintPen.setWidth(1);

	areaSize = QSizeF(0.0,0.0);
	grid.clear();
}

// SLOT: Set the grid spacing
void GLEGrid::setGrid(QPointF newGrid)
{
	if (gridSpacing != newGrid)
	{
		gridSpacing = newGrid;
		updateGrid();
	}
}

// SLOT: Set the pixmap area
void GLEGrid::setArea(QSize area)
{
	if (area != areaSize)
	{
		areaSize = area;
		updateGrid();
	}
}

// SLOT: set the resolution
void GLEGrid::setDPI(double new_dpi)
{
	if (dpi != new_dpi)
	{
		dpi = new_dpi;
		updatePen();
		updateGrid();
	}
}

// Update the pen width according to the resolution
void GLEGrid::updatePen()
{
	//paintPen.setWidth((int) floor(dpi/100.0));
	paintPen.setWidth(1);
}

QPen GLEGrid::pen()
{
	return(paintPen);
}

// Update the grid points if something has changed
void GLEGrid::updateGrid()
{
	// Only set up the grid if we have the spacings and dpi
	if ((gridSpacing.x() == 0.0) || (gridSpacing.y() == 0.0) || (dpi == 0.0) || (areaSize.height() == 0))
	{
		return;
	}

	// Clear the grid
	grid.clear();
	// Get the grid spacing in QT units
	QPointF qtGrid = QGLE::relGLEToQt(gridSpacing,dpi);

	// Create the grid
	for(double i = (GS_OFFSET)*dpi; i < areaSize.width() ; i += qtGrid.x())
	{
		for(double j = (GS_OFFSET)*dpi; j < areaSize.height() ; j += qtGrid.y())
		{
			grid.append(QPointF(i,areaSize.height() - j));			
		}
	}
}

// Return the grid for displaying
QList<QPointF> GLEGrid::allPoints()
{
	return(grid);
}

// Find the nearest point to a given (QT) point: used for snap to grid
QPointF GLEGrid::nearestPoint(QPointF qt, double *distance)
{
	double shortest_distance = 1e6;
	int shortest_index;
	double dist = 1e6;

	for (int i=0;i<grid.size();i++)
	{
		// Calculate distance between qt and grid.at(i)
		dist = sqrt(pow(grid.at(i).x()-qt.x(),2) + pow(grid.at(i).y() - qt.y(),2));
		if (dist < shortest_distance)
		{
			shortest_distance = dist;
			shortest_index = i;
		}
	}

	if (distance)
		*distance = dist;

	return(grid.at(shortest_index));
}
		

