/***********************************************************************************
 * QGLE - A Graphical Interface to GLE                                             *
 * Copyright (C) 2006  A. S. Budden & J. Struyf                                    *
 *                                                                                 *
 * This program is free software; you can redistribute it and/or                   *
 * modify it under the terms of the GNU General Public License                     *
 * as published by the Free Software Foundation; either version 2                  *
 * of the License, or (at your option) any later version.                          *
 *                                                                                 *
 * This program is distributed in the hope that it will be useful,                 *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of                  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                   *
 * GNU General Public License for more details.                                    *
 *                                                                                 *
 * You should have received a copy of the GNU General Public License               *
 * along with this program; if not, write to the Free Software                     *
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA. *
 *                                                                                 *
 * Also add information on how to contact you by electronic and paper mail.        *
 ***********************************************************************************/

/***************************************************************
 * mainwindow.cpp: The class implementation for GLEMainWindow. *
 ***************************************************************/

#include <QtGui>
#include <QtDebug>
#include <math.h>

#include "mainwindow.h"
#include "qgle_statics.h"
#include "settings.h"
#include "settings_dialogue.h"
#include "qgs.h"
#include "about.h"
#include "newfile.h"
// #include "downloader.h"

//! Class Constructor
GLEMainWindow::GLEMainWindow(int argc, char *argv[])
{
	// Resize to an initial default size
	resize(500,300);

	// Set the Window Icon
	setWindowIcon(QIcon(":images/gle.png"));
		
	// Coordinate display default
	coordView = CoordinateCart;
	lastPoint = QPointF(0,0);

	// Some variable initialisation
	autoScale = false;
	reRenderOnEdit = false;
	imageWithExtras = false;
	waitForRender = false;

	// Initialise temp file names
	QString tempDir = QDir::tempPath();
	QRegExp rxEndSlash(".*[\\\\/]$");
	if (!rxEndSlash.exactMatch(tempDir))
		tempDir += QDir::separator();
	tempFiles[CombinedEPS] = tempDir + "qgle_combined.eps";
	tempFiles[SimpleEPS] = tempDir + "qgle.eps";
	// CombinedGLE will be added on the fly.

	// Get the location of the qgle executable
	// -> used to check if gle is installed in the same location
	exeLocation = QGLE::GetDirName(QGLE::GetExeName("qgle"));
	GLEGhostScriptLocation = "?";
	GsLibVersionNumber = "?";
	GLEVersionNumber = "?";
		
	// Create the drawing area in a scroll widget.  For high resolutions,
	// this may be undesirable as the whole image must be rendered by GS
	drawingArea = new GLEDrawingArea(this);
	scrollArea = new QScrollArea;
	scrollArea->setWidget(drawingArea);
	setCentralWidget(scrollArea);

	// Display a title on the title bar
	setWindowTitle(QString("%1 %2")
			.arg(APP_NAME)
			.arg(QGLE_VERSION));

	// Create the component parts
	createActions();
	createMenus();
	createStatusBar();
	createToolBars();
	createEditModeToolBars();

	// Read the application settings
	settings = new GLESettings(this);
	connect(settings, SIGNAL(polarSnapStartAngleChanged(double)),
			drawingArea, SLOT(setPolarSnapStartAngle(double)));
	connect(settings, SIGNAL(polarSnapIncAngleChanged(double)),
			drawingArea, SLOT(setPolarSnapIncAngle(double)));
	settings->readAll();

	// We always resize, we just only save the size if requested
	resize(settings->size());
	move(settings->position());

	// Initialise snapping
	if (settings->polarSnapOnStart())
		polarSnapAct->setChecked(true);
	if (settings->gridSnapOnStart())
		gridSnapAct->setChecked(true);
	if (settings->osnapOnStart())
		osnapAct->setChecked(true);
	if (settings->orthoSnapOnStart())
		orthoSnapAct->setChecked(true);

	
	if (!checkGLE()) exit(-1);

	// As necessary, start the file monitor
	fileMonitorTimer = new QTimer(this);
	connect(fileMonitorTimer, SIGNAL(timeout()),
			this, SLOT(checkForFileUpdates()));
	updateFileMonitor(settings->monitorOpenFile());
	

	// The following line is important as it allows
	// QImage objects to be passed by the signal-slot
	// mechanism.
	qRegisterMetaType<QImage>("QImage");
	qRegisterMetaType<GLEFileInfo>("GLEFileInfo");

	// Initialise the serverRunning variable
	serverRunning = false;	

	// Start in preview mode
	currentMode = GLESettings::PreviewMode;
	oldMode = currentMode;

	serverInPreviewMode = true;
	serverAct->setChecked(true);
	toggleByFunction = true;
	previewModeAct->setChecked(true);
	toggleByFunction = false;

	// Setting connections
	
	// If the grid is changed, update the drawing
	connect(settings, SIGNAL(gridChanged(QPointF)),
			drawingArea, SLOT(setGrid(QPointF)));
	// If the file monitor is switched on or off, 
	// sort this out
	connect(settings, SIGNAL(monitorFileChanged(bool)),
			this, SLOT(updateFileMonitor(bool)));

	// If a tool is selected, notify the drawing
	connect(this, SIGNAL(toolSelected(int)),
			drawingArea, SLOT(setTool(int)));
	// If a new image is rendered, update the drawing
	connect(this, SIGNAL(imageChanged(QImage)),
			drawingArea, SLOT(updateDisplay(QImage)));
	// If a point is drawn, set the base point used for
	// relative coordinates
	connect(drawingArea, SIGNAL(basePointChanged(QPointF)),
			this, SLOT(setRelativeBasePoint(QPointF)));
	// If the drawing area requests a status bar message,
	// display it
	connect(drawingArea, SIGNAL(updateStatusBar(QString)),
			this, SLOT(statusBarMessage(QString)));
	// If the dirty flag is set or unset, enable or disable
	// the 'save' button accordingly
	connect(drawingArea, SIGNAL(dirtyFlagChanged(bool)),
			this, SLOT(setSaveEnable(bool)));
	// If the dpi changes, update the status bar display
	connect(drawingArea, SIGNAL(dpiChanged(double)),
			this, SLOT(updateDPIDisplay(double)));

	// Get the default resolution
	drawingArea->setDPI((double) settings->dpi());

	// Update the grid spacing
	drawingArea->setGrid(settings->grid());

	// Accept keyboard events
	setFocusPolicy(Qt::StrongFocus);
	setFocus(Qt::OtherFocusReason);
	
	// Updates to port?
	
	// Drag and drop
	setAcceptDrops(true);

	// Check for arguments
	QRegExp rx = QGLE::fileRegExp();

	for(int i=0;i<argc;i++)
	{
		if (rx.exactMatch(QString(argv[i])))
		{
			openFile(rx.capturedTexts()[1]);
			break;
		}
	}

		
}

// Switch on or off the file monitor
void GLEMainWindow::updateFileMonitor(bool state)
{
	if (state)
	{
		if (fileMonitorTimer->isActive())
			return;
		// One second timer
		fileMonitorTimer->start(1000);
	}
	else
	{
		if (!fileMonitorTimer->isActive())
			return;
		fileMonitorTimer->stop();
	}
	// used to wait another "tick" before auto-reloading
	fileMonitorCount = 0;
}

bool GLEMainWindow::askAnywayBeforeReload() {
	// if there are objects created, ask the user anyway
	// unless the user has checked "always include objects"
	return currentFile.isGLE() && 
	       drawingArea->thereAreObjects() && 
	       settings->askAboutObjects();
}

// Check whether the file has changed
void GLEMainWindow::checkForFileUpdates()
{
	int answer;
	if (currentFile.hasChanged())
	{
		if (settings->monitorAutoReloadFile() && !askAnywayBeforeReload())
		{
			if (fileMonitorCount < 1)
			{
				// wait another "tick" to make sure file saved correctly
				// don't want to load partially saved files
				fileMonitorCount++;
				return;
			}
		}
		// reset counter for next iteration
		fileMonitorCount = 0;
		
		// If the file has changed, update the stored time stamp and
		// stop the timer while we ask the user what to do.
		fileMonitorTimer->stop();

		// Ask the user whether to reload and, if we have any objects, should
		// we keep them?
		QString msg = QString(tr("File %1 has changed")).arg(currentFile.primaryFile());
		if (settings->monitorAutoReloadFile() && !askAnywayBeforeReload())
		{
			answer = 1;
		}
		else if (currentFile.isGLE() && drawingArea->thereAreObjects())
		{
			if (settings->askAboutObjects())
				answer = QMessageBox::question(this, tr("%1 File Monitor").arg(APP_NAME),
						msg, tr("Reload file"),
						tr("Reload keeping objects"),
						tr("Ignore"),
						0,2);
			else
			{
				// still ask user to reload file (?)
				// but don't bother him with choice about objects
				answer = QMessageBox::question(this, tr("%1 File Monitor").arg(APP_NAME),
					msg, tr("Reload file"),
					tr("Ignore"),
					QString(),
					0,1);
				if (answer == 0) answer = 1;
			}

		}
		else if (currentFile.isEPS() || currentFile.isGLE())
		{
			answer = QMessageBox::question(this, tr("%1 File Monitor").arg(APP_NAME),
					msg, tr("Reload file"),
					tr("Ignore"),
					QString(),
					0,1);
		}
		else
			return;

		switch(answer)
		{
			case 0:
				qDebug() << "Reload file selected";
				// Get rid of the objects, switch to preview mode
				// and draw the changed file

				oldMode = currentMode;
				//saveMode = false;
				qDebug() << "2: Storing mode: " << oldMode;
				//previewModeAct->setChecked(true);

				resetDrawing();
				if (currentFile.isGLE())
				{
					currentFile.readGLECode();
					renderGLE();
				}
				else
					renderEPS();
				break;

				//saveMode = true;
			case 1:
				// Reload file keeping objects
				qDebug() << "Reload file keeping objects selected";

				oldMode = currentMode;
				qDebug() << "3: Storing mode: " << oldMode;
//				previewModeAct->setChecked(true);

				// Render the GLE or EPS file appropriately depending
				// on the current mode
				if (currentFile.isGLE())
				{
					//saveMode = false;
					currentFile.readGLECode();
					waitForRender = true;
					qDebug() << "Performing first render";

					if (currentMode == GLESettings::PreviewMode)
					{
						renderGLEandObjects();
						reRenderOnEdit = true;
					}
					else
					{
						renderGLE();
						drawingArea->setNewObjectsFlag();
					}
				}
				else
					renderEPS();
				break;

			case 2:
				qDebug() << "Ignore selected";
				break;
		}

		// Restart the timer
		currentFile.updateTimeStamp();
		fileMonitorTimer->start();
	}

}

// Accept URI lists that have been dragged over the application
void GLEMainWindow::dragEnterEvent(QDragEnterEvent *event)
{
	if (event->mimeData()->hasFormat("text/uri-list"))
		event->acceptProposedAction();
}

// Handle dropped URIs
void GLEMainWindow::dropEvent(QDropEvent *event)
{
	// This will need to check that the file is a GLE or EPS
	// file and then try to open it, by stripping off the file:// part
	// and calling openFile(fileName).  Worth checking that the file
	// exists and is readable while we're at it.

	// We're only interested in the first URI as this isn't a 
	// multi-document interface
	QString uri = event->mimeData()->urls().at(0).toString();

	QRegExp rx = QGLE::fileRegExp();
	qDebug() << "Pattern: " << rx.pattern();

	if (rx.exactMatch(uri))
	{
		qDebug() << "Dropped file: " << 
			uri << ": " << rx.capturedTexts()[2];
		QGLE::flushIO();
		openFile(rx.capturedTexts()[2]);
	}
	else
		QMessageBox::information(this, 
				"Drop Event", 
				"Dropped file cannot be opened",
				QMessageBox::Ok);


	event->acceptProposedAction();
}

// User pressed a key
// Keyboard shortcuts are based on those used in AutoCAD
void GLEMainWindow::keyPressEvent(QKeyEvent *event)
{
	// Since we haven't any keyboard shortcuts that
	// use modifiers, if there are any modifiers,
	// ignore the key press
	if (event->modifiers() != Qt::NoModifier)
		event->ignore();
	
	switch (event->key())
	{
		// Delete or escape keys are handled by the drawing itself
		case Qt::Key_Delete:
		case Qt::Key_Escape:
			drawingArea->hitKey(event->key(), event->modifiers());
			event->accept();
			break;

		case Qt::Key_F9:
			// F9 is used to toggle grid snap
			gridSnapAct->toggle();
			event->accept();
			break;

		case Qt::Key_F3:
			// F3 is used to toggle OSNAP: snapping to objects
			osnapAct->toggle();
			event->accept();
			break;

		case Qt::Key_F10:
			// F10 is used to toggle polar snap: customisable snapping
			polarSnapAct->toggle();
			event->accept();
			break;

		case Qt::Key_F8:
			// F8 is used to toggle ortho snap: 0 or 90 degree lines
			orthoSnapAct->toggle();
			event->accept();
			break;

		case Qt::Key_F7:
			// F7 is used to toggle grid visibility
			gridAct->toggle();
			event->accept();
			break;

		case Qt::Key_F1:
			// F1 shows the about box
			aboutAct->trigger();
			event->accept();
			break;
			
		case Qt::Key_Equal:
		case Qt::Key_Plus:
			// The equals key is used to zoom in
			// as is plus.
			zoomInAct->trigger();
			event->accept();
			break;

		case Qt::Key_Minus:
			// The minus key is used to zoom out
			zoomOutAct->trigger();
			event->accept();
			break;

		case Qt::Key_F6:
			// F6 cycles through the various coordinate views
			switch (coordView)
			{
				case CoordinateOff:
					coordView = CoordinateCart;
					mousePosition->setVisible(true);
					break;
				case CoordinateCart:
					coordView = CoordinatePolar;
					mousePosition->setVisible(true);
					break;
				case CoordinatePolar:
					mousePosition->setVisible(true);
					coordView = CoordinateRelCart;
					break;
				case CoordinateRelCart:
					mousePosition->setVisible(true);
					coordView = CoordinateRelPolar;
					break;
				case CoordinateRelPolar:
					mousePosition->setVisible(false);
					coordView = CoordinateOff;
					break;
			}
			updateMousePosition();
			event->accept();
			break;

		case Qt::Key_Shift:
			qDebug() << "Shift key depressed";
			drawingArea->modifierToggle(GLEDrawingArea::ShiftKey, true);
			break;

		case Qt::Key_Control:
			qDebug() << "Control key depressed";
			drawingArea->modifierToggle(GLEDrawingArea::CtrlKey, true);
			break;

		case Qt::Key_Alt:
			qDebug() << "Alt key depressed";
			drawingArea->modifierToggle(GLEDrawingArea::AltKey, true);
			break;



		default:
			event->ignore();
	}

}

void GLEMainWindow::keyReleaseEvent(QKeyEvent *event)
{
	// Since we haven't any keyboard shortcuts that
	// use modifiers, if there are any modifiers,
	// ignore the key press
	if (event->modifiers() != Qt::NoModifier)
		event->ignore();
	
	switch (event->key())
	{
		case Qt::Key_Shift:
			qDebug() << "Shift key released";
			drawingArea->modifierToggle(GLEDrawingArea::ShiftKey, false);
			break;

		case Qt::Key_Control:
			qDebug() << "Control key released";
			drawingArea->modifierToggle(GLEDrawingArea::CtrlKey, false);
			break;

		case Qt::Key_Alt:
			qDebug() << "Alt key released";
			drawingArea->modifierToggle(GLEDrawingArea::AltKey, false);
			break;

		default:
			event->ignore();
	}
}


// Toggle the status of the server
void GLEMainWindow::serverToggle(bool switchOn)
{
	if (switchOn && (!serverRunning))
		startServer();
	else if ((!switchOn) && serverRunning)
		stopServer();
}

// Start the server
void GLEMainWindow::startServer()
{
	qDebug() << "Starting server";
	// Initialise the GS library
	initLibGS();


	// Remember to update the server as well if it's changed anywhere else
	serverThread = new GLEServerThread(this, settings->port());

	// Connect server signals
	connect(serverThread, SIGNAL(serverMessage(QString)),
			this, SLOT(updateServerStatus(QString)));
	
	// Notify the drawing area when the image is ready for display
	connect(serverThread, SIGNAL(imageReady(GLEFileInfo)),
			this, SLOT(newFileInfo(GLEFileInfo)));

	// Notify the rendering code that an image is ready
	connect(serverThread, SIGNAL(readyToGo(QImage)),
			this, SLOT(renderComplete(QImage)));

	// Notify drawing area of resolution changes from the server
	connect(serverThread, SIGNAL(dpiChanged(double)),
			drawingArea, SLOT(setDPI(double)));

	// Allow EPS files to be rendered
	connect(this, SIGNAL(newEPS(QString, double, bool)),
			serverThread, SLOT(EPSToImage(QString, double, bool)));

	// Update changes to autoscale
	connect(this, SIGNAL(autoScaleChanged(bool)),
			serverThread, SLOT(setUserAutoScale(bool)));

	// Update changes to default resolution
	connect(this, SIGNAL(defaultResolutionChanged(double)),
			serverThread, SLOT(setDefaultResolution(double)));
	
	// Initialise the server
	serverThread->setDPI(drawingArea->getDPI());
	serverThread->setUserAutoScale(settings->autoScaleOnOpen());
	serverThread->setDefaultResolution((double) settings->dpi());
	serverThread->initialise();

	// Wait for the server to start - DANGER of infinite loop
	// while(!serverThread->isListening());
	serverThread->waitForServer();
	
	serverRunning = true;
}

// A new file has been opened/called, set the window title 
// and update the display
void GLEMainWindow::newFileInfo(GLEFileInfo newFile)
{
	// Use QFileInfo for comparison to iron out
	// any string differences such as use of
	// backslashes or forward slashes, included /../
	// strings etc.
	QFileInfo oldFileInfo(currentFile.gleFile());
	QFileInfo newFileInfo(newFile.gleFile());

	if (oldFileInfo == newFileInfo)
	{
		// Copy the file into memory
		currentFile = newFile; // needed to copy the new image
		// Read the GLE code
		currentFile.readGLECode();

		// If there are drawing objects, ask whether to keep them
		if (drawingArea->thereAreObjects())
		{
			int answer = QMessageBox::Yes;
			if (settings->askAboutObjects())
			{
				QString msg = tr("Include objects already drawn?");
				answer = QMessageBox::question(this, tr("GLE file updated"),
						msg,
						QMessageBox::Yes,
						QMessageBox::No,
						0);
			}

			// If the user wants to keep the objects, render appropriately
			if (answer == QMessageBox::Yes)
			{
				if (currentMode == GLESettings::PreviewMode)
					renderGLEandObjects();
				else
				{
					drawingArea->setNewObjectsFlag();
					emit imageChanged(currentFile.image());
				}
				return;
			}
		}

		// If the user doesn't want to keep the objects, delete them and update
		// the display
		resetDrawing();
		emit imageChanged(currentFile.image());
		return;

	}

	// File is new, should we save the old one?
	if (drawingArea->isDirty())
	{
		int reply = QMessageBox::question(this, tr("Open new GLE file"),
				tr("Save changes to drawing?"),
				QMessageBox::Yes,
				QMessageBox::No,
				QMessageBox::Cancel);

		if (reply == QMessageBox::Yes)
			save();
		else if (reply == QMessageBox::No)
		{
			drawingArea->clearDirty();
		}
		else
		{
			return;
		}
	}

	// Clear the drawing objects
	resetDrawing();

	// Read the file
	currentFile = newFile;
	currentFile.readGLECode();

	QFileInfo file;
	file.setFile(currentFile.gleFile());
	if (file.isReadable())
	{
		settings->setPwd(file.dir().path());
	}
	else
	{
		file.setFile(currentFile.epsFile());
		if (file.isReadable())
			settings->setPwd(file.dir().path());
	}
	
	setWindowTitle(QString("%1 %2 - ")
			.arg(APP_NAME)
			.arg(QGLE_VERSION) + 
			currentFile.gleFile() + 
			"[*]");

	// Update the display
	emit imageChanged(currentFile.image());
}


// Stop the server
void GLEMainWindow::stopServer()
{
	// This should clean it all up in theory
	delete(serverThread);
	serverRunning = false;
}

// Initialise GS
void GLEMainWindow::initLibGS() 
{
	QString error;
	bool found = false;
	QString location = settings->getLibGSLocation();
	GSLibFunctions* fct = GSLibFunctions::getInstance();
	
	// First try to use the same version of GhostScript as GLE does
	if (fct->loadLibrary(location, error) != 0)
	{
		location = GLEGhostScriptLocation;
		location += QGLE::gsLibFileName();
	}
	else 
	{
		found = true;
	}
	
	// If this does not work, consult the user
	while (!found && fct->loadLibrary(location, error) != 0) 
	{
		QString err = tr("%1 was unable to load the GhostScript library."
				"\n\nError: '%1'.\n\nClick 'Browse' to locate %2"
				".\n\nNote: you can install Ghostscript from:\n"
				"http://www.cs.wisc.edu/~ghost/")
			.arg(APP_NAME)
			.arg(error)
			.arg(QGLE::gsLibFileName());
		if (QMessageBox::warning(this, APP_NAME, 
			err,
			tr("Browse"),
			tr("Abort"), 0, 0, 1) == 1) 
		{
				exit(0);
		}
		location = QFileDialog::getOpenFileName(this, tr("Locate the GhostScript library"), "/",
				QGLE::libraryFilter()
		);
	}
	// GSLibFunctions might discover the real location
	settings->setLibGSLocation(fct->libGSLocation());
	GsLibVersionNumber = fct->getVersion();
}

// Find GLE
bool GLEMainWindow::checkGLE()
{
	QString error;
	QString location = settings->GLELocation();
	QString msg;
	bool found = false;
	
	// First check same location of QGLE executable
	if (GLEVersion(location).isEmpty())
	{
		location = exeLocation;
		location += QGLE::gleExecutableName();
	}
	else
	{
		found = true;
	}
	
	// Check for a valid version string from GLE
	while(!found && GLEVersion(location).isEmpty())
	{
		// Ask the user to locate the gle executable
		msg = tr("Please click 'Browse' to locate the GLE executable: %1.")
			.arg(QGLE::gleExecutableName());

		if(QMessageBox::warning(this, APP_NAME, msg,
			tr("Browse"),
			tr("Abort"), 0, 0, 1) == 1)
		{
			return(false);
		}
		location = QFileDialog::getOpenFileName(this, tr("Locate the GLE executable"), "/",
				QGLE::executableFilter());
	}

	// Save the gle executables location and exit
	settings->setGLELocation(location);

	return(true);	
}

// Run GLE and return the version string
QString GLEMainWindow::GLEVersion(QString gleCommand)
{
	QRegExp rxVer("GLE version\\:(.*)");
	QRegExp rxGS("GhostScript\\:(.*)");

	GLEVersionNumber = "";	
	GLEGhostScriptLocation = "?";

	QProcess gle;
	qDebug() << "Running GLE:" << gleCommand;
	gle.start(gleCommand, QStringList() << "-info");
	// Allow 2 seconds of problems!
	if(!gle.waitForStarted(2000))
		qDebug() << "Couldn't start GLE";
	if(!gle.waitForFinished(2000))
		qDebug() << "GLE Didn't finish";

	// Get the STDERR and read into a QString
	QString output = gle.readAllStandardOutput();

	// Scan each line for "GLE version ..."
	if (!output.isEmpty())
	{
		QString line;
		QStringList lines;
		lines = output.split(QRegExp("[\\r\\n]+"));
		foreach(line, lines)
		{
			if(rxVer.exactMatch(line))
				GLEVersionNumber = rxVer.capturedTexts()[1].trimmed();
			if(rxGS.exactMatch(line))
				GLEGhostScriptLocation = rxGS.capturedTexts()[1].trimmed();
		}
	}
	
	if (!GLEVersionNumber.isEmpty()) {
		qDebug() << "Version:" << GLEVersionNumber;
		qDebug() << "GhostScript location:" << GLEGhostScriptLocation;
	}
	
	// Return an empty string if it failed
	return GLEVersionNumber;
}

// Called when QGLE is closed
void GLEMainWindow::closeEvent(QCloseEvent *event)
{

	// Check whether to save changes
	if (drawingArea->isDirty())
	{
		int reply = QMessageBox::question(this, tr("Quit QGLE"),
				tr("Save changes to drawing?"),
				QMessageBox::Yes,
				QMessageBox::No,
				QMessageBox::Cancel);

		if (reply == QMessageBox::Yes)
			save();
		else if (reply == QMessageBox::No)
		{
			drawingArea->clearDirty();
		}
		else
		{
			event->ignore();
			return;
		}
	}

	// If the window size and position should be saved, update them
	if (settings->storeSize())
	{
		settings->setPosition(pos());
		settings->setSize(size());
	}
	// Store the settings
	settings->writeAll();

	event->accept();
}

void GLEMainWindow::createActions()
{
	// Create the actions for each menu entry and connect them
	// to an appropriate slot
	
	// Quit the application
	quitAct = new QAction(tr("&Quit"), this);
	quitAct->setShortcut(Qt::Key_Q | Qt::ControlModifier);
	quitAct->setStatusTip(tr("Exit QGLE"));
	connect(quitAct, SIGNAL(triggered()),
			this, SLOT(close()));

	// Create a new file
	newAct = new QAction(QIcon(":images/new.png"), tr("&New"), this);
	newAct->setShortcut(Qt::Key_N | Qt::ControlModifier);
	newAct->setStatusTip(tr("Create a new file"));
	connect(newAct, SIGNAL(triggered()),
			this, SLOT(newFile()));

	// Open a specified file
	openAct = new QAction(QIcon(":images/open.png"), tr("&Open"), this);
	openAct->setShortcut(Qt::Key_O | Qt::ControlModifier);
	openAct->setStatusTip(tr("Open an existing file"));
	connect(openAct, SIGNAL(triggered()),
			this, SLOT(openFile()));

	// Save the file
	saveAct = new QAction(QIcon(":images/save.png"), tr("&Save"), this);
	saveAct->setShortcut(Qt::Key_S | Qt::ControlModifier);
	saveAct->setStatusTip(tr("Save changes"));
	saveAct->setEnabled(false);
	connect(saveAct, SIGNAL(triggered()),
			this, SLOT(save()));

	// Save the file under a new name
	saveAsAct = new QAction(tr("Save &As"), this);
	saveAsAct->setStatusTip(tr("Save under a new name"));
	connect(saveAsAct, SIGNAL(triggered()),
			this, SLOT(saveAs()));

	// Show an about box
	aboutAct = new QAction(tr("&About"), this);
	aboutAct->setStatusTip(tr("Show information about QGLE"));
	connect(aboutAct, SIGNAL(triggered()),
			this, SLOT(about()));

	// Toggle the grid visibility
	gridAct = new QAction(QIcon(":images/grid.png"), tr("Grid"), this);
	gridAct->setStatusTip(tr("Toggle the grid on and off"));
	gridAct->setCheckable(true);
	gridAct->setChecked(false);
	connect(gridAct, SIGNAL(toggled(bool)),
			drawingArea, SLOT(gridToggle(bool)));

	// Toggle grid snap
	gridSnapAct = new QAction(QIcon(":images/grid_snap.png"), tr("Snap Grid"), this);
	gridSnapAct->setStatusTip(tr("Toggle grid snap on and off"));
	gridSnapAct->setCheckable(true);
	gridSnapAct->setChecked(false);
	connect(gridSnapAct, SIGNAL(toggled(bool)),
			drawingArea, SLOT(gridSnapToggle(bool)));

	// Toggle osnap
	osnapAct = new QAction(QIcon(":images/osnap.png"), tr("Snap to Objects"), this);
	osnapAct->setStatusTip(tr("Toggle OSNAP on and off"));
	osnapAct->setCheckable(true);
	osnapAct->setChecked(false);
	connect(osnapAct, SIGNAL(toggled(bool)),
			drawingArea, SLOT(osnapToggle(bool)));
	
	// Toggle orthoSnap
	orthoSnapAct = new QAction(QIcon(":images/orthosnap.png"), tr("Snap to axes"), this);
	orthoSnapAct->setStatusTip(tr("Toggle ORTHO on and off"));
	orthoSnapAct->setCheckable(true);
	orthoSnapAct->setChecked(false);
	connect(orthoSnapAct, SIGNAL(toggled(bool)),
			drawingArea, SLOT(orthoSnapToggle(bool)));

	// Toggle polar snap
	polarSnapAct = new QAction(QIcon(":images/polarsnap.png"), tr("Polar snap"), this);
	polarSnapAct->setStatusTip(tr("Toggle customisable snaps on and off"));
	polarSnapAct->setCheckable(true);
	polarSnapAct->setChecked(false);
	connect(polarSnapAct, SIGNAL(toggled(bool)),
			drawingArea, SLOT(polarSnapToggle(bool)));

	// Toggle whether the server is running
	serverAct = new QAction(QIcon(":images/server.png"), tr("Server"), this);
	serverAct->setStatusTip(tr("Start or stop the server"));
	serverAct->setCheckable(true);
	serverAct->setChecked(false);
	connect(serverAct, SIGNAL(toggled(bool)),
			this, SLOT(serverToggle(bool)));

	// Switch to preview mode
	previewModeAct = new QAction(QIcon(":images/preview_mode.png"), tr("Preview Mode"), this);
	previewModeAct->setStatusTip(tr("Switch to preview mode"));
	previewModeAct->setCheckable(true);
	previewModeAct->setChecked(false);
	connect(previewModeAct, SIGNAL(toggled(bool)),
			this, SLOT(previewModeToggle(bool)));

	// Switch to edit mode
	editModeAct = new QAction(QIcon(":images/edit_mode.png"), tr("Edit Mode"), this);
	editModeAct->setStatusTip(tr("Switch to edit mode"));
	editModeAct->setCheckable(true);
	editModeAct->setChecked(false);
	connect(editModeAct, SIGNAL(toggled(bool)),
			this, SLOT(editModeToggle(bool)));

	// Toggle whether the server is running
	editorAct = new QAction(QIcon(":images/editor.png"), tr("Open in Editor"), this);
	editorAct->setStatusTip(tr("Open file in text editor"));
	editorAct->setCheckable(false);
	connect(editorAct, SIGNAL(triggered()), this, SLOT(openInTextEditor()));
	
	// Zoom In
	zoomInAct = new QAction(QIcon(":images/zoom_in.png"), tr("Zoom In"), this);
	zoomInAct->setStatusTip(tr("Zoom In"));
	zoomInAct->setCheckable(false);
	connect(zoomInAct, SIGNAL(triggered()),
			this, SLOT(zoomIn()));

	// Zoom Out
	zoomOutAct = new QAction(QIcon(":images/zoom_out.png"), tr("Zoom Out"), this);
	zoomOutAct->setStatusTip(tr("Zoom Out"));
	zoomOutAct->setCheckable(false);
	connect(zoomOutAct, SIGNAL(triggered()),
			this, SLOT(zoomOut()));

	// Tools
	
	// Pointer tool
	pointerToolAct = new QAction(QIcon(":images/pointer.png"), tr("Pointer Tool"), this);
	pointerToolAct->setStatusTip(tr("Selection Tool"));
	pointerToolAct->setCheckable(true);
	pointerToolAct->setChecked(true);
	connect(pointerToolAct, SIGNAL(toggled(bool)),
			this, SLOT(pointerToolToggle(bool)));

	// Line tool
	lineToolAct = new QAction(QIcon(":images/line.png"), tr("Line Tool"), this);
	lineToolAct->setStatusTip(tr("Line Tool"));
	lineToolAct->setCheckable(true);
	lineToolAct->setChecked(false);
	connect(lineToolAct, SIGNAL(toggled(bool)),
			this, SLOT(lineToolToggle(bool)));

	// Circle tool
	circleToolAct = new QAction(QIcon(":images/circle.png"), tr("Circle Tool"), this);
	circleToolAct->setStatusTip(tr("Circle Tool"));
	circleToolAct->setCheckable(true);
	circleToolAct->setChecked(false);
	connect(circleToolAct, SIGNAL(toggled(bool)),
			this, SLOT(circleToolToggle(bool)));

	// Arc tool
	arcToolAct = new QAction(QIcon(":images/arc_3p.png"), tr("Arc Tool"), this);
	arcToolAct->setStatusTip(tr("Circular Arc Tool"));
	arcToolAct->setCheckable(true);
	arcToolAct->setChecked(false);
	connect(arcToolAct, SIGNAL(toggled(bool)),
			this, SLOT(arcToolToggle(bool)));

	// move tool
	moveToolAct = new QAction(QIcon(":images/amove.png"), tr("Move Tool"), this);
	moveToolAct->setStatusTip(tr("Final Point Tool"));
	moveToolAct->setCheckable(true);
	moveToolAct->setChecked(false);
	connect(moveToolAct, SIGNAL(toggled(bool)),
			this, SLOT(moveToolToggle(bool)));

	// Display the settings dialogue box
	settingsAct = new QAction(tr("&Options"), this);
	settingsAct->setStatusTip(tr("Configure QGLE"));
	connect(settingsAct, SIGNAL(triggered()),
			this, SLOT(openSettingsDialogue()));
	
	// Test the download function
	downloadAct = new QAction(tr("&Download"), this);
	downloadAct->setStatusTip(tr("Download GhostScript libgs.so"));
	connect(downloadAct, SIGNAL(triggered()),
			this, SLOT(openDownloadDialogue()));
}

// Create the toolbars
void GLEMainWindow::createToolBars()
{

	fileToolBar = addToolBar(tr("File"));
	fileToolBar->addAction(newAct);
	fileToolBar->addAction(openAct);
	fileToolBar->addAction(saveAct);

	modeToolBar = addToolBar(tr("Mode"));
//	modeToolBar->addAction(serverAct);
	modeToolBar->addAction(previewModeAct);
	modeToolBar->addAction(editModeAct);

	viewToolBar = addToolBar(tr("View"));
	viewToolBar->addAction(zoomInAct);
	viewToolBar->addAction(zoomOutAct);
	viewToolBar->addAction(editorAct);

	// Initialise toolsToolBar and controlToolBar to 0
	toolsToolBar = 0;
	controlToolBar = 0;
}

void GLEMainWindow::createEditModeToolBars()
{
	toolsToolBar = new QToolBar(tr("Tools"));
	toolsToolBar->setOrientation(Qt::Vertical);
	toolsToolBar->setAllowedAreas(Qt::LeftToolBarArea | Qt::RightToolBarArea);
	toolsToolBar->addAction(pointerToolAct);
	toolsToolBar->addAction(moveToolAct);
	toolsToolBar->addAction(lineToolAct);
	toolsToolBar->addAction(circleToolAct);
	toolsToolBar->addAction(arcToolAct);
	addToolBar(Qt::LeftToolBarArea, toolsToolBar);

	controlToolBar = new QToolBar(tr("Drawing Control"));
	controlToolBar->setOrientation(Qt::Vertical);
	controlToolBar->setAllowedAreas(Qt::LeftToolBarArea | Qt::RightToolBarArea);
	controlToolBar->addAction(gridAct);
	controlToolBar->addAction(gridSnapAct);
	controlToolBar->addAction(osnapAct);
	controlToolBar->addAction(orthoSnapAct);
	controlToolBar->addAction(polarSnapAct);
	addToolBar(Qt::LeftToolBarArea, controlToolBar);

}

void GLEMainWindow::enableEditModeToolBars(bool enable)
{
	if (!enable)
	{
		// Hide first, next gray out
		toolsToolBar->hide();
		controlToolBar->hide();
	}
	toolsToolBar->setEnabled(enable);
	controlToolBar->setEnabled(enable);
	if (enable)
	{
		// Enable first, then show
		toolsToolBar->show();
		controlToolBar->show();
	}
}

void GLEMainWindow::destroyEditModeToolBars()
{
	if (toolsToolBar)
	{
		removeToolBar(toolsToolBar);
		delete toolsToolBar;
		toolsToolBar = 0;
	}
	if (controlToolBar)
	{
		removeToolBar(controlToolBar);
		delete controlToolBar;
		controlToolBar = 0;
	}
}

// SLOT: user clicked the preview mode button
void GLEMainWindow::previewModeToggle(bool state)
{
	// User switched mode on
	if (state)
	{
		// Switch to preview mode
		currentMode = GLESettings::PreviewMode;

		toggleByFunction = true;
		editModeAct->setChecked(false);
		toggleByFunction = false;
		drawingArea->setEditMode(false);

		// Hide the tools toolbar
		enableEditModeToolBars(false);

		// Select no tool
		emit toolSelected(GLEDrawingArea::NoTool);

		// call renderGLEandObjects and then
		// emit imageChanged with currentFile's resulting
		// complete image object (imageWithExtras())
		QFileInfo gleFileInfo(currentFile.gleFile());
		// Only process if new objects have been created
		// or if objects have been moved
		if (drawingArea->thereAreNewObjects())
		{
			qDebug() << "Rendering new objects";
			QGLE::flushIO();
			renderGLEandObjects();

			if (settings->saveOnPreview())
				save();
			drawingArea->clearNewObjectsFlag();
		}
		else
		{
			emit imageChanged(currentFile.imageWithExtras());
		}
	}
	else
	{
		if(!toggleByFunction)
			previewModeAct->setChecked(true);
	}
}


// SLOT: user clicked the edit mode button
void GLEMainWindow::editModeToggle(bool state)
{
	// User switched mode on
	if (state)
	{
		if (currentFile.gleFile().isEmpty())
		{
			toggleByFunction = true;
			editModeAct->setChecked(false);
			previewModeAct->setChecked(true);
			toggleByFunction = false;
			return;
		}

		currentMode = GLESettings::EditMode;
		drawingArea->setEditMode(true);

		toggleByFunction = true;
		previewModeAct->setChecked(false);
		toggleByFunction = false;
		enableEditModeToolBars(true);

		if (pointerToolAct->isChecked())
			emit toolSelected(GLEDrawingArea::PointerTool);
		else if (lineToolAct->isChecked())
			emit toolSelected(GLEDrawingArea::LineTool);
		else if (circleToolAct->isChecked())
			emit toolSelected(GLEDrawingArea::CircleTool);
		else if (arcToolAct->isChecked())
			emit toolSelected(GLEDrawingArea::ArcTool);
		else if (moveToolAct->isChecked())
			emit toolSelected(GLEDrawingArea::AMoveTool);

		// We should probably only emit this when there is
		// definitely an image, but edit mode shouldn't be
		// available if there isn't one
		if (reRenderOnEdit)
		{
			QFileInfo fi(currentFile.gleFile());
			if (fi.isReadable())
				renderGLE();
			else
			{
				double dpi = drawingArea->getDPI();
				QSizeF qtSize = QGLE::relGLEToQt(newDiagramSize, dpi);
				qtSize.setWidth(qtSize.width() + 2*GS_OFFSET*dpi);
				qtSize.setHeight(qtSize.height() + 2*GS_OFFSET*dpi);

				QImage image = QImage(qtSize.toSize(),QImage::Format_RGB32);
				image.fill(QColor(Qt::white).rgb());
				currentFile.setImage(image,drawingArea->getDPI());
				emit imageChanged(image);
			}
		}
		else
			emit imageChanged(currentFile.image());
	}
	else
	{
		if(!toggleByFunction)
			editModeAct->setChecked(true);
	}
}


// SLOT: user clicked the pointer tool button
void GLEMainWindow::pointerToolToggle(bool state)
{
	// User switched mode on
	if (state)
	{
		toggleByFunction = true;
		lineToolAct->setChecked(false);
		circleToolAct->setChecked(false);
		arcToolAct->setChecked(false);
		moveToolAct->setChecked(false);
		toggleByFunction = false;

		// Notify drawing area
		emit toolSelected(GLEDrawingArea::PointerTool);
	}
	else
	{
		if(!toggleByFunction)
			pointerToolAct->setChecked(true);
	}
}

// SLOT: user clicked the line tool button
void GLEMainWindow::lineToolToggle(bool state)
{
	// User switched mode on
	if (state)
	{
		toggleByFunction = true;
		pointerToolAct->setChecked(false);
		moveToolAct->setChecked(false);
		circleToolAct->setChecked(false);
		arcToolAct->setChecked(false);
		toggleByFunction = false;

		// Notify drawing area
		emit toolSelected(GLEDrawingArea::LineTool);
	}
	else
	{
		if(!toggleByFunction)
			lineToolAct->setChecked(true);
	}
}

// SLOT: user clicked the circle tool button
void GLEMainWindow::circleToolToggle(bool state)
{
	// User switched mode on
	if (state)
	{
		toggleByFunction = true;
		pointerToolAct->setChecked(false);
		moveToolAct->setChecked(false);
		lineToolAct->setChecked(false);
		arcToolAct->setChecked(false);
		toggleByFunction = false;

		// Notify drawing area
		emit toolSelected(GLEDrawingArea::CircleTool);
	}
	else
	{
		if(!toggleByFunction)
			circleToolAct->setChecked(true);
	}
}

// SLOT: user clicked the arc tool button
void GLEMainWindow::arcToolToggle(bool state)
{
	// User switched mode on
	if (state)
	{
		toggleByFunction = true;
		pointerToolAct->setChecked(false);
		moveToolAct->setChecked(false);
		lineToolAct->setChecked(false);
		circleToolAct->setChecked(false);
		toggleByFunction = false;

		// Notify drawing area
		emit toolSelected(GLEDrawingArea::ArcTool);
	}
	else
	{
		if(!toggleByFunction)
			arcToolAct->setChecked(true);
	}
}

// SLOT: user clicked the pointer tool button
void GLEMainWindow::moveToolToggle(bool state)
{
	// User switched mode on
	if (state)
	{
		toggleByFunction = true;
		lineToolAct->setChecked(false);
		circleToolAct->setChecked(false);
		arcToolAct->setChecked(false);
		pointerToolAct->setChecked(false);
		toggleByFunction = false;
		
		// Notify drawing area
		emit toolSelected(GLEDrawingArea::AMoveTool);
	}
	else
	{
		if(!toggleByFunction)
			moveToolAct->setChecked(true);
	}
}

void GLEMainWindow::renderEPS()
{
	// Render an EPS file from currentFile.epsFile()
	renderEPS(currentFile.epsFile());
}

// Render an EPS file from a given QString
void GLEMainWindow::renderEPS(QString epsFile)
{
	QImage img;

	// Get the dpi to communicate to the server thread
	double dpi = drawingArea->getDPI();

	QFileInfo epsFileInfo(epsFile);
	
	qDebug() << "About to render " << epsFileInfo.absoluteFilePath();
	
	emit newEPS(epsFileInfo.absoluteFilePath(), dpi, autoScale);

}

// A signal from the server to indicate that rendering is complete
void GLEMainWindow::renderComplete(QImage image)
{
	qDebug() << "Render Complete";
	// If we're doing a RenderGLEandObjects, set the appropriate
	// image in the fileinfo
	if (imageWithExtras)
	{
		currentFile.setImageWithExtras(image);
		imageWithExtras = false;
	}
	// Otherwise set the main image and set up the renderer so 
	// that it will rerender when going into preview mode
	else
	{
		currentFile.setImage(image, drawingArea->getDPI());
		drawingArea->setNewObjectsFlag();
	}
	// Update the display
	emit imageChanged(image);

	// Set the window title appropriately
	if (currentFile.gleFile().isEmpty())
		setWindowTitle(QString("%1 %2 - ")
				.arg(APP_NAME)
				.arg(QGLE_VERSION) +
				currentFile.epsFile() + "[*]");
	else
		setWindowTitle(QString("%1 %2 - ")
				.arg(APP_NAME)
				.arg(QGLE_VERSION) +
				currentFile.gleFile() + "[*]");

	waitForRender = false;
}

void GLEMainWindow::renderGLE()
{
	// Render a GLE file from currentFile.gleFile()
	renderGLE(currentFile.gleFile());
}

void GLEMainWindow::renderGLE(QString gleFile, bool combinedOutput)
{
	// Render a GLE file into a temporary eps and
	// return the resulting image
	QString tempFile;
	if (combinedOutput)
		tempFile = tempFiles[CombinedEPS];
	else
		tempFile = tempFiles[SimpleEPS];

	QString gleCommand;

//	if(!checkGLE())
//		return;

	// Find the location of GLE
	gleCommand = settings->GLELocation();

	qDebug(tempFile.toLatin1().constData());
	qDebug(gleFile.toLatin1().constData());

	// Create some worker objects
	QFileInfo gleFileInfo(gleFile);
	QProcess gle;

	// Set the working directory to the directory of the gle file
	gle.setWorkingDirectory(gleFileInfo.dir().path());

	// Run GLE
	gle.start(gleCommand, QStringList() << "-d" << "eps" << "-o" << tempFile << gleFile);
	gle.waitForStarted(2000);

	gle.waitForFinished(5000);

	QString result = gle.readAll();

	qDebug() << result;
	qDebug() << "Exit code: " << gle.exitCode();

	// Now we should really check to see if it worked or not!
	if (gle.exitCode() == 0)
	{
		qDebug() << "Successful run of GLE, rendering EPS:" << tempFile;

		// Store EPS filename
		if (!combinedOutput)
			currentFile.setEpsFile(tempFile);

		// Render the eps
		renderEPS(tempFile);

		// Now we should really remove the tempFile
	}
	else
	{
		qDebug() << "Error running GLE";
		QMessageBox::critical(this, tr("GLE Error"), result,
				QMessageBox::Ok,
				QMessageBox::NoButton,
				QMessageBox::NoButton);
	}
}


void GLEMainWindow::renderGLEandObjects()
{
	// Combine GLE code in gleFile() and objects in drawing area
	// and render.  Requires creation of temporary files.
	// Read the resulting image into setImageWithExtras
	
	// temp Directory is QString QDir::tempPath()
	// use standardised filename for now: qgle_combined.gle
	
	// Now the temp file is created in the directory where the GLE file is

	QFileInfo gleOrig(currentFile.gleFile());

	QString tempFile = gleOrig.absolutePath() + QDir::separator() + "qgle_combined.gle";
	tempFiles[CombinedGLE] = tempFile;

	// Clear the tempfile if it is there (we'd better hope that the user
	// doesn't have an important file called "qgle_combined.gle"...
	if(QFile::exists(tempFile))
		QFile::remove(tempFile);
			
	// Open the temp file for writing
	QFile gle(tempFile);
	if(!gle.open(QIODevice::Truncate | QIODevice::WriteOnly | QIODevice::Text))
	{
		QErrorMessage *errMsg = new QErrorMessage(this);
		errMsg->showMessage(tr("Couldn't write to temporary file"));
		return;
	}

	// Get the base file code
	QStringList baseCode = currentFile.gleCode();

	// Write it to the file
	for(int i=0;i<baseCode.size();i++)
		gle.write(baseCode.at(i).toUtf8());

	// Start with a "restoredefaults" command to ensure
	// that parameters aren't affected
	gle.write(QString("\nrestoredefaults\n").toUtf8());

	// Get the object drawing code and write that to a file
	QStringList gleCode = drawingArea->gleObjects();
	for(int i=0;i<gleCode.size();i++)
	{
		gle.write(gleCode.at(i).toUtf8().constData());
		gle.write("\n");
	}

	// Close the file
	gle.close();
	

	// Render
	imageWithExtras = true;
	renderGLE(tempFile, true);

	// Delete the file
	gle.remove();
	
}




// SLOT: display the about box
void GLEMainWindow::about()
{
	box = new AboutBox;
	box->setModal(true);
	box->show();
}

// Create the menus and add the actions
void GLEMainWindow::createMenus()
{
	// File menu
	fileMenu = menuBar()->addMenu(tr("&File"));
	fileMenu->addAction(newAct);
	fileMenu->addAction(openAct);
	fileMenu->addAction(saveAct);
	fileMenu->addAction(saveAsAct);
	fileMenu->addSeparator();
	fileMenu->addAction(quitAct);

	// Tools menu
	toolsMenu = menuBar()->addMenu(tr("&Tools"));
	toolsMenu->addAction(settingsAct);
	// toolsMenu->addAction(downloadAct);

	// Help menu
	helpMenu = menuBar()->addMenu(tr("&Help"));
	helpMenu->addAction(aboutAct);
}

#define STATUS_BAR_LABEL_STYLE QFrame::NoFrame | QFrame::Plain

void GLEMainWindow::createStatusBar()
{
	// Create a status bar with an initial message
	statusBar()->showMessage(tr("Ready"));

	// Create a coordinate display (using GLE coordinates)
	mousePosition = new QLabel();
	mousePosition->setFrameStyle(STATUS_BAR_LABEL_STYLE);
	mousePosition->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	dpiDisplay = new QLabel();
	dpiDisplay->setFrameStyle(STATUS_BAR_LABEL_STYLE);
	dpiDisplay->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	// Create the snap button
/*	gridSnapButton = new QPushButton(tr("SNAP"));
	gridSnapButton->resize(0.4*gridSnapButton->size());
	gridSnapButton->setFlat(true);
	gridSnapButton->setCheckable(true);
	connect(gridSnapButton, SIGNAL(toggled(bool)),
			gridSnapAct, SLOT(setChecked(bool)));
	statusBar()->addPermanentWidget(gridSnapButton);*/

	// Set size and disable auto resize?
	statusBar()->addPermanentWidget(dpiDisplay);
	statusBar()->addPermanentWidget(mousePosition);
	updateMousePosition(QPointF(0.0,0.0));
	// Display the default resolution
	updateDPIDisplay(-1.0);
	connect(drawingArea, SIGNAL(mouseMoved(QPointF)),
			this, SLOT(updateMousePosition(QPointF)));
}

void GLEMainWindow::updateDPIDisplay(double newDPI)
{
	// Display the current resolution on the status bar
	if (newDPI < 0.0)
	{
		dpiDisplay->setVisible(false);
	}
	else
	{
		dpiDisplay->setVisible(true);
		dpiDisplay->setText(QString(tr("R:%1", "Resolution Indicator"))
				.arg(newDPI,8,'f',2));
		dpiDisplay->setFixedSize(100,24);
	}
}


// Update the coordinate display
void GLEMainWindow::updateMousePosition(QPointF gle)
{

	if (gle == QPointF(-1.0,-1.0))
		gle = lastMousePosition;
	else
		lastMousePosition = gle;

	double r, theta;
	switch (coordView)
	{
		case CoordinateOff:
			// Don't display anything
			mousePosition->setText("");
			break;
		case CoordinateCart:
			// Cartesian coordinates
			mousePosition->setText(QString("%1,%2").arg(gle.x(),5,'f',2).arg(gle.y(),5,'f',2));
			break;
		case CoordinatePolar:
			// Polar coordinates
			r = sqrt(pow(gle.x(),2)+pow(gle.y(),2));
			theta = QGLE::radiansToDegrees(atan2(gle.y(),gle.x()));
			mousePosition->setText(QString("%1<%2").arg(r,5,'f',2).arg(theta,5,'f',2));
			break;
		case CoordinateRelCart:
			// Cartesian coordinates relative to the last point used in drawing
			mousePosition->setText(QString("@%1,%2")
					.arg(gle.x()-lastPoint.x(),5,'f',2)
					.arg(gle.y()-lastPoint.y(),5,'f',2));
			break;
		case CoordinateRelPolar:
			// Polar coordinates relative to the last point used in drawing
			r = sqrt(pow(gle.x()-lastPoint.x(),2)+pow(gle.y()-lastPoint.y(),2));
			theta = QGLE::radiansToDegrees(atan2(gle.y()-lastPoint.y(),gle.x()-lastPoint.x()));
			mousePosition->setText(QString("@%1<%2").arg(r,5,'f',2).arg(theta,5,'f',2));
			break;
	}
	// Fix the size of the display
	mousePosition->setFixedSize(100,24);

}

// SLOT: Display a requested message on the status bar
void GLEMainWindow::updateServerStatus(QString message)
{
	statusBar()->showMessage(message);
}

// SLOT: Display a requested message on the status bar
void GLEMainWindow::statusBarMessage(QString msg)
{
	statusBar()->showMessage(msg);
}

// SLOT: Open the Settings Dialogue
void GLEMainWindow::openSettingsDialogue()
{
	SettingsDialogue *settingsDialogue = new SettingsDialogue(this);
	connect(settingsDialogue, SIGNAL(autoScaleChanged(bool)),
			this, SIGNAL(autoScaleChanged(bool)));
	connect(settingsDialogue, SIGNAL(defaultResolutionChanged(double)),
			this, SIGNAL(defaultResolutionChanged(double)));
	//settingsDialogue->setModal(true);
	settingsDialogue->exec();

}

void GLEMainWindow::openDownloadDialogue()
{
/*	QGLEDownloader* down = new QGLEDownloader(this);
	down->setModal(true);
	down->show();
	down->exec();*/
}

void GLEMainWindow::save()
{
	if (drawingArea->isDirty())
	{
		if (currentFile.hasFileName())
		{
			QFileInfo file;
			file.setFile(currentFile.gleFile());
			if (file.isReadable() && file.isWritable())
			{
				save(currentFile.gleFile());
				return;
			}
		}
		// If we don't have a file name and/or the file
		// isn't readable and writable, ask for a new file
		// name.
		saveAs();
	}
}

void GLEMainWindow::saveAs()
{
	// Ask for a filename and save accordingly
	QString fileName = QFileDialog::getSaveFileName(this,
			tr("Choose a file name to save under"),
			settings->pwd(),
			tr("GLE Files (*.gle)"));

	if (fileName.isEmpty())
		return;
	
	// Add a .gle if there isn't one already
	QRegExp rxGLE(".*\\.[gG][lL][eE]");
	if(!rxGLE.exactMatch(fileName))
		fileName += ".gle";
	QFileInfo fname(fileName);
	settings->setPwd(fname.dir().path());
	save(fileName);

}

void GLEMainWindow::save(QString fileName)
{
	// Save to a given filename
	QFile dest(fileName);
	bool wasActive = false;

	// Stop the timer as it'll get confused by us writing out own file
	if (fileMonitorTimer->isActive())
	{
		wasActive = true;
		fileMonitorTimer->stop();
	}

	// Can we open the output file?
	if (!dest.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Truncate))
	{
		QErrorMessage *errMsg = new QErrorMessage(this);
		errMsg->showMessage(tr("Couldn't open output file"));
		if (wasActive)
			fileMonitorTimer->start();
		return;
	}

	// Get the base code
	QStringList baseCode = currentFile.gleCode();

	// Write to disk
	for(int i=0;i<baseCode.size();i++)
		dest.write(baseCode.at(i).toUtf8());

	// Ensure default parameters are present at the start
	dest.write(QString("\nrestoredefaults\n").toUtf8());
	

	// Get the drawing objects code
	QStringList extraCode = drawingArea->gleObjects();

	// Write to disk
	for(int i=0;i<extraCode.size();i++)
	{
		dest.write(extraCode.at(i).toUtf8());
		dest.write("\n");
	}

	// Clear the dirty flag
	drawingArea->clearDirty();
	// Check that currentFile points to the correct file
	// (primarily for "save as")
	currentFile.setGleFile(fileName);
	setWindowTitle(QString("%1 %2 - ")
			.arg(APP_NAME)
			.arg(QGLE_VERSION) + 
			fileName + "[*]");
	
	// Close the output and update the time stamp
	dest.close();
	currentFile.updateTimeStamp();

	// Restart the timer if appropriate
	if (wasActive)
		fileMonitorTimer->start();
}

// SLOT: Open an existing file
void GLEMainWindow::openFile(QString fileName)
{
	// Stop the timer
	bool wasActive = false;
	if (fileMonitorTimer->isActive())
	{
		wasActive = true;
		fileMonitorTimer->stop();
	}

	// Check whether to autoscale and get the resolution
	autoScale = settings->autoScaleOnOpen();
	drawingArea->setDPI((double) settings->dpi());
	serverThread->setDPI(drawingArea->getDPI());

	// If changes have been made, ask what to do
	if (drawingArea->isDirty())
	{
		int reply = QMessageBox::question(this, tr("Open New File"),
				tr("Save changes to drawing?"),
				QMessageBox::Yes,
				QMessageBox::No,
				QMessageBox::Cancel);

		if (reply == QMessageBox::Yes)
			save();
		else if (reply == QMessageBox::Cancel)
			return;
	}

	// If we don't have a filename (i.e. if called as a slot rather than
	// as a function), ask what file to open
	if (fileName.isEmpty())
		fileName = QFileDialog::getOpenFileName(this,
				tr("Select a file to open"),
				settings->pwd(),
				tr("GLE Files (*.gle);;EPS Files (*.eps)"));

	// If we now have a filename, open it
	if (!fileName.isEmpty())
	{
		// Get the file information
		QFileInfo fname(fileName);
		settings->setPwd(fname.dir().path());
		qDebug() << "Open file: " << fileName;
		// Regular expressions to check whether it's an EPS
		// file or a GLE file
		QRegExp rxeps(".*\\.eps$");
		QRegExp rxgle(".*\\.gle$");

		// Prepare for autoscale
		emit autoScaleChanged(settings->autoScaleOnOpen());
		
		// An EPS file, so open it.
		if(rxeps.exactMatch(fileName))
		{
			currentFile.clear();
			currentFile.setEpsFile(fileName);
			currentFile.updateTimeStamp();
			resetDrawing();
			previewModeAct->setChecked(true);
			renderEPS(fileName);
			
		}
		// A GLE file, so render it and open the resulting
		// EPS file
		else if (rxgle.exactMatch(fileName))
		{
			currentFile.clear();
			currentFile.setGleFile(fileName);
			currentFile.readGLECode();
			currentFile.updateTimeStamp();
			resetDrawing();
			previewModeAct->setChecked(true);
			renderGLE();
		}
		else
		{
		}
	}

	autoScale = false;
	
	// Restart the timer if appropriate
	if (wasActive)
		fileMonitorTimer->start();

}

// SLOT: Create a blank page
void GLEMainWindow::newFile()
{
	// Should we save changes?
	if (drawingArea->isDirty())
	{
		int reply = QMessageBox::question(this, tr("Create New File"),
				tr("Save changes to drawing?"),
				QMessageBox::Yes,
				QMessageBox::No,
				QMessageBox::Cancel);

		if (reply == QMessageBox::Yes)
			save();
		else if (reply == QMessageBox::Cancel)
			return;
	}

	// Ask the user what size drawing to create
	NewFileBox nfb;

	int dialogue_result = nfb.exec();

	if (dialogue_result == QDialog::Rejected)
		return;

	newDiagramSize = nfb.size();

	// Clear the current drawing
	currentFile.clear();
	
	// Autoscale new drawing
	if (settings->autoScaleOnOpen()) {
		QSize size = getScrollAreaSize();
		int dpi = QGLE::computeAutoScaleDPIFromCm(size, newDiagramSize.width(), newDiagramSize.height());
		settings->setDPI(dpi);
		serverThread->setDPI(dpi);
	}
	
	// Convert the GLE size to QT coordinates
	QSizeF qtSize = QGLE::relGLEToQt(newDiagramSize,
				settings->dpi());

	// Add a bit (in the same way that GLE does)
	qtSize.setWidth(qtSize.width() + 2*GS_OFFSET*settings->dpi());
	qtSize.setHeight(qtSize.height() + 2*GS_OFFSET*settings->dpi());

	// Create a blank image of the correct size
	QImage image = QImage(qtSize.toSize(),QImage::Format_RGB32);
	// Fill the image in white
	image.fill(QColor(Qt::white).rgb());

	// We don't have a filename yet
	currentFile.setGleFile(tr("No Name"));
	currentFile.setHasFileName(false);
	// The file is very simple: just one line "size x y"
	QStringList code;
	code.append(QString("size %1 %2")
				.arg(QGLE::GLEToStr(newDiagramSize.width()))
				.arg(QGLE::GLEToStr(newDiagramSize.height())));
	currentFile.setGLECode(code);

	// Both images (with and without objects) are blank
	currentFile.setImage(image, drawingArea->getDPI());
	currentFile.setImageWithExtras(image);
	// Set the window title
	setWindowTitle(QString("%1 - ")
			.arg(APP_NAME)
			.arg(QGLE_VERSION) + 
			currentFile.gleFile() + "[*]");
	resetDrawing();
	// Since we've added a "size" line, the file has (arguably) changed
	drawingArea->setDirty();

	// Update the display
	emit imageChanged(image);

	// One assumes that edit mode will be wanted for a new file
	editModeAct->setChecked(true);

}

void GLEMainWindow::resetDrawing()
{
	// Reset the drawing.
	drawingArea->clearDirty();
	drawingArea->clearObjects();
	drawingArea->clearNewObjectsFlag();
}

// SLOT: Zoom In
void GLEMainWindow::zoomIn()
{
	double dpi = drawingArea->getDPI();
	// There appears to be a problem with RenderGLEandObjects here
	dpi *= ZOOM_STEP;
	drawingArea->setDPI(dpi);
	zoom(dpi);
}

// SLOT: Zoom Out
void GLEMainWindow::zoomOut()
{
	double dpi = drawingArea->getDPI();
	dpi /= ZOOM_STEP;
	drawingArea->setDPI(dpi);
	zoom(dpi);
}

// Zoom worker
void GLEMainWindow::zoom(double dpi)
{
	QFileInfo fi(currentFile.epsFile());
	// If the file is readable, re-render it in the new scale
	if (currentMode == GLESettings::PreviewMode)
	{
		reRenderOnEdit = true;
		renderGLEandObjects();
	}
	else
	{
		if (fi.isReadable())
		{
			renderEPS();
		}
		// Otherwise, it must be a 'new drawing', so recreate from scratch
		else
		{
			QSizeF qtSize = QGLE::relGLEToQt(newDiagramSize,
					drawingArea->getDPI());
			qtSize.setWidth(qtSize.width() + 2*GS_OFFSET*dpi);
			qtSize.setHeight(qtSize.height() + 2*GS_OFFSET*dpi);

			QImage image = QImage(qtSize.toSize(),QImage::Format_RGB32);
			image.fill(QColor(Qt::white).rgb());
			currentFile.setImage(image, drawingArea->getDPI());

			// Set the new objects flag so that it'll be rerendered
			// when switching to preview mode
			drawingArea->setNewObjectsFlag();
			// Update the display
			emit imageChanged(image);
		}
	}

}

void GLEMainWindow::openInTextEditor() 
{
	QString file;
	bool res;

	file = currentFile.primaryFile();
	QFileInfo fh(file);
	if (!fh.isReadable())
	{
		int answer = QMessageBox::question(this, APP_NAME,
				tr("File does not exist, do you wish to save the current document?"),
				tr("Yes"),
				tr("Cancel"), 0, 1);
		if (answer == 0)
			save();
		else
			return;
	}

	file = currentFile.primaryFile();
	fh.setFile(file);

	if (!fh.isReadable())
	{
		QMessageBox::critical(this, APP_NAME, tr("Problem saving!"),
				QMessageBox::Ok,
				QMessageBox::NoButton,
				QMessageBox::NoButton);
		return;
	}


	QString editor = settings->editorLocation();
	if (editor.isEmpty())
	{
		QMessageBox::critical(this, APP_NAME, tr("Please select text editor in Options | Tools"),
				QMessageBox::Ok,
				QMessageBox::NoButton,
				QMessageBox::NoButton);
	} 
	else
	{	
		QProcess edit;
#ifdef Q_WS_WIN
		file.replace('/', '\\');
#endif		
		qDebug() << "Opening:" << file;

#ifdef Q_WS_MAC
		// This checks whether we're on a Mac and if so, handles .app files appropriately
		QRegExp rxApp(".*\\.app");
		if (rxApp.exactMatch(editor))
		{
			res = edit.startDetached("open", QStringList() << "-a" << editor << file);
		}
		else
#endif
		{
			res = edit.startDetached(editor, QStringList() << file);
		}

// TEMPORARY FIX for problem with startDetached always failing
#ifndef Q_WS_MAC
		if (!res) 
		{
			QMessageBox::critical(this, APP_NAME, tr("Failed to start text editor (configure it in Options | Tools)"),
				QMessageBox::Ok,
				QMessageBox::NoButton,
				QMessageBox::NoButton);
		}
#endif
	}
}

// Set a point upon which to base the relative coordinate system
void GLEMainWindow::setRelativeBasePoint(QPointF gle)
{
	lastPoint = gle;
}

// Get the size of the scoll area
QSize GLEMainWindow::getScrollAreaSize()
{
	return scrollArea->size();
}

// Get the current file name
QString GLEMainWindow::getCurrentGleFile()
{
	return currentFile.gleFile();
}

// Enable or disable the save button
void GLEMainWindow::setSaveEnable(bool state)
{
	saveAct->setEnabled(state);
}

