/***************************************************************************
 *   Copyright (C) 2005 by Piotr Szymanski <niedakh@gmail.com>             *
 *                                                                         *
 * This library is free software; you can redistribute it and/or           *
 * modify it under the terms of the GNU Lesser General Public              *
 * License as published by the Free Software Foundation; either            *
 * version 2.1 of the License, or (at your option) any later version.      *
 *                                                                         *
 * This library is distributed in the hope that it will be useful,         *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * Lesser General Public License for more details.                         *
 *                                                                         *
 * You should have received a copy of the GNU Lesser General Public        *
 * License along with this library; if not, write to the Free Software     *
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,                  *
 * MA  02110-1301  USA                                                     *
 ***************************************************************************/

#include <math.h>
#include <qpixmap.h>
#include <qpainter.h>
#include <qstring.h>
#include <qrect.h>
#include "qgs.h"

#define MINIMUM(a, b)    ((a) < (b) ? (a) : (b))

namespace GSApiWrapper
{
// C API wrappers
int GSDLLCALL handleStdin(void *caller_handle, char *buf, int len)
{
	if(caller_handle != 0)
		return static_cast <GSInterpreterLib*>(caller_handle) -> gs_input(buf,len);
	return 0;
}

int GSDLLCALL handleStdout(void *caller_handle, const char *str, int len)
{
	if(caller_handle != 0)
	return static_cast <GSInterpreterLib*>(caller_handle) -> gs_output(str,len);
	return 0;
}

int GSDLLCALL handleStderr(void *caller_handle, const char *str, int len)
{
	if(caller_handle != 0)
		return static_cast <GSInterpreterLib*>(caller_handle) -> gs_error(str,len);
	return 0;
}

int open(void* handle, void* device)
{
	if (handle != 0) return static_cast <GSInterpreterLib*>(handle) -> open(device);
	return 0;
}

int preclose(void * handle, void * device )
{
	if (handle != 0) return static_cast <GSInterpreterLib*>(handle) -> preclose(device);
	return 0;
}

int close(void * handle, void * device )
{
	if (handle != 0) return static_cast <GSInterpreterLib*>(handle) -> close(device);
	return 0;
}


int presize(void * handle, void * device, int width, int height,
        int raster, unsigned int format)
{
	// qDebug("presize called, width/height/raster: %d/%d/%d",width,height,raster);
	if (handle != 0) return static_cast <GSInterpreterLib*>(handle) -> presize(device, width, height, raster, format);
	return 0;
}

int size(void *handle, void *device, int width, int height, int raster, unsigned int format, unsigned char *pimage)
{
	// qDebug("size called, width/height/raster: %d/%d/%d",width,height,raster);
	if (handle != 0) return static_cast <GSInterpreterLib*>(handle) -> size (device, width, height, raster, format, pimage);
	return 0;
}

int sync(void * handle, void * device  )
{
	// qDebug("sync called");
	if (handle != 0) return static_cast <GSInterpreterLib*>(handle) -> sync(device);
	return 0;
}

int page(void *handle, void * device, int copies, int flush)
{
	// qDebug("page called: copies/flush = %d/%d",copies,flush);
	if (handle != 0) return static_cast <GSInterpreterLib*>(handle) -> page (device, copies, flush);
	return 0;
}

int update(void * handle, void * device, int x, int y, int w, int h)
{
	if (handle != 0)
	{
		GSInterpreterLib* tmp=static_cast <GSInterpreterLib*>(handle);
		// check it here, because this function is often executed by libgs, we dont want to waste performance
		if (tmp->progressive())
		{
			// qDebug("update called: (%d,%d)x(%d,%d)",x,y,w,h);
			return static_cast <GSInterpreterLib*>(handle) -> update(device, x, y, w, h);
		}
	}
	return 0;
}

int separation(void * handle, void * device, int comp_num, const char * name, unsigned short c, unsigned short m, unsigned short y, unsigned short k)
{
	// qDebug("separation called");
	if (handle != 0) return static_cast <GSInterpreterLib*>(handle) -> separation(device, comp_num, name, c,m,y,k);
	return 0;
}

#ifdef QGS_USE_MAJOR_V2

// Newer versions of GhostScript support "separation", but we don't need this

typedef display_callback display_callback_qgs;
display_callback_qgs device = {
	sizeof ( display_callback_qgs ),
	DISPLAY_VERSION_MAJOR,
	DISPLAY_VERSION_MINOR,
	&open,
	&preclose,
	&close,
	&presize,
	&size,
	&sync,
	&page,
	&update,
	NULL,
	NULL,
	&separation
};

#else

// For now use older version of structure, which is compatible with GhostScript 8.15
// This version is shipped with Fedora Core 5

typedef struct display_callback_v1_s display_callback_qgs;

display_callback_qgs device = {
	sizeof ( display_callback_qgs ),
	DISPLAY_VERSION_MAJOR_V1,
	DISPLAY_VERSION_MINOR_V1,
	&open,
	&preclose,
	&close,
	&presize,
	&size,
	&sync,
	&page,
	&update,
	NULL,
	NULL
};
#endif

}

/********* GSInterpreterLib ************/

GSInterpreterLib::GSInterpreterLib() :
	m_running(false),
	m_sync(false),
	m_syncDone(false),
	m_cnvmode(QGS_DISPLAY_32),
	m_conv(NULL),
	m_orientation(0),
	m_magnify(1.0),
	m_width(0),
	m_height(0),
	m_dpi(100.0),
	m_media(QString::null),
	m_Gwidth(0),
	m_Gheight(0),
	m_imageChar(0),
	m_format(0),
	m_raster(0),
	m_textaa(1),
	m_graphaa(1),
	m_progr(false),
	m_pfonts(true),
	m_display(true),
	m_wasSize(false),
	m_wasPage(false),
	m_img(0),
	m_argsCCount(0),
	m_argsChar(0)
{
	m_gs = GSLibFunctions::getInstance();
	if (!m_gs->isLoaded()) return;

	int exit = m_gs->gsapi_new_instance((void**)&ghostScriptInstance,this);
	// qDebug("GS instance created with code: %d",exit);
	if (exit && !handleExit(exit)) return;
	
	GSLibFunctions* fct = GSLibFunctions::getInstance();
	if (fct->getVersionMajor() < 8) {
		/* 32 bit does not work with GhostView 7.04 */
		m_cnvmode = QGS_DISPLAY_24;
		qDebug("GhostScript in 24 mode");
	} else {
		qDebug("GhostScript in 32 mode");
	}
}

GSInterpreterLib::~GSInterpreterLib()
{
	if (running())	m_gs->gsapi_exit(ghostScriptInstance);
	if (m_argsChar)
	{
		for (int i=0;i<m_argsCCount;i++) delete [] *(m_argsChar+i);
		delete [] m_argsChar;
	}
	if (m_gs->isLoaded()) m_gs->gsapi_delete_instance(ghostScriptInstance);
	if (m_conv != NULL) delete[] m_conv;
	delete m_img;
}

// interpreter state functions

bool GSInterpreterLib::start(bool setStdio)
{
	m_sync=false;
	// qDebug("setting m_sync to %d in start",m_sync);

	if ( setStdio )
	{
		m_gs->gsapi_set_stdio(ghostScriptInstance, &(GSApiWrapper::handleStdin),
		                                           &(GSApiWrapper::handleStdout),
		                                           &(GSApiWrapper::handleStderr));
	}

	// qDebug("setting display");
	int call = m_gs->gsapi_set_display_callback(ghostScriptInstance, (display_callback*)&(GSApiWrapper::device));
	if (call && !handleExit(call)) return false;
	// qDebug("converting args to char**");
	argsToChar();
	// qDebug("setting args");
	call = m_gs->gsapi_init_with_args (ghostScriptInstance,m_argsCCount,m_argsChar);
	if (call && !handleExit(call)) return false;
	QString set;
	set.sprintf("<< /Orientation %d >> setpagedevice .locksafe",m_orientation);
	QByteArray strdata = set.toLatin1();
	m_gs->gsapi_run_string_with_length (ghostScriptInstance,strdata.constData(),set.length(),0,&call);
	m_running = handleExit ( call ) ;
	return m_running;
}

bool GSInterpreterLib::stop()
{
	if (m_running)
	{
		int exit = m_gs->gsapi_exit(ghostScriptInstance);
		if (exit) handleExit(exit);
		// qDebug("Ghostscript stopped in stop with code %d",exit);
		m_running = false;
		m_sync = false;
		// qDebug("setting m_sync to %d in stop", m_sync);
	}
	return m_running;
}

// set options
void GSInterpreterLib::setGhostscriptArguments( const QStringList &list )
{
	if ( m_args != list )
	{
		m_args=list;
		stop();
	}
}

void GSInterpreterLib::setOrientation( int orientation )
{
	if( m_orientation != orientation )
	{
		m_orientation = orientation;
		stop();
	}
}

void GSInterpreterLib::setMagnify( double magnify )
{
	if( m_magnify != magnify )
	{
		m_magnify = magnify;
		stop();
	}
}

void GSInterpreterLib::setMedia( const QString &media )
{
	if( m_media != media )
	{
		m_media = media;
		stop();
	}
}

void GSInterpreterLib::setSize( int w, int h )
{
	if ( m_width != w )
	{
		m_width=w;
		stop();
	}
	if ( m_height != h )
	{
		m_height=h;
		stop();
	}
}

void GSInterpreterLib::setDPI( double dpi )
{
	if ( m_dpi != dpi )
	{
		m_dpi=dpi;
		stop();
	}
}

void GSInterpreterLib::setDisplay( bool display )
{
	if( m_display != display )
	{
		m_display = display;
		stop();
	}
}

void GSInterpreterLib::setPlatformFonts( bool pfonts )
{
	if( m_pfonts != pfonts )
	{
		m_pfonts = pfonts;
		stop();
	}
}

void GSInterpreterLib::setProgressive( bool progr )
{
	if( m_progr != progr )
	{
		m_progr = progr;
		stop();
	}
}

void GSInterpreterLib::setAABits(int text, int graphics )
{
	if( m_textaa != text )
	{
		m_textaa = text;
		stop();
	}
	if( m_graphaa != graphics )
	{
		m_graphaa = graphics;
		stop();
	}
}


void GSInterpreterLib::setBuffered( bool buffered )
{
	if( m_buffered != buffered ) m_buffered = buffered;
}

void GSInterpreterLib::setOrigin(double ox, double oy)
{
	m_OriginX = ox;
	m_OriginY = oy;
}

bool GSInterpreterLib::run(FILE * tmp, Position pos, bool sync)
{
	if (!m_gs->isLoaded()) return false;
	if (fseek(tmp,pos.first, SEEK_SET)) return false;
	if (!running()) start();
	m_syncDone=false;
	m_sync=sync;
	m_wasPage=false;
	// qDebug("setting m_sync to %d in run", m_sync);
	char buf [4096];
	int exit_code, left=pos.second - pos.first;
	if (progressive() && m_sync )
	{
		// qDebug("Making image Gw/Gh %d/%d",m_Gwidth,m_Gheight);
		if ( m_img != 0 ) qDebug("Possible memory leak, m_img is initialised before run");
		m_img=new QImage(m_Gwidth, m_Gheight, QImage::Format_RGB32);
		m_img->invertPixels();
		emit(Started(m_img));
	}
	m_gs->gsapi_run_string_begin (ghostScriptInstance, 0, &exit_code);
	if (exit_code && !handleExit(exit_code)) return false;
	if (m_OriginX != 0.0 || m_OriginY != 0.0) {
		// Add a translate if bounding box not at origin zero (to support LaTeX processed output)
		QString str = QString("%1 %2 translate\n").arg(-m_OriginX).arg(-m_OriginY);
		QByteArray strdata = str.toLatin1();
		m_gs->gsapi_run_string_continue(ghostScriptInstance, strdata.constData(), str.length(), 0, &exit_code);
		if (exit_code && !handleExit(exit_code)) return false;
	}
	// qDebug("Left to write: %d",left);
	while (left > 0) {
		int toread = sizeof(buf)-1;
		if (left < toread) toread = left;
		int read = fread (buf, sizeof(char), toread, tmp);
		// Make sure to add trailing zero to buffer
		buf[read] = 0;
		if (read == 0) {
			// Make sure to exit if nothing more in file
			break;
		}
		m_gs->gsapi_run_string_continue (ghostScriptInstance, buf, read, 0, &exit_code);
		if (exit_code && !handleExit(exit_code)) return false;
		left -= read;
	}
	m_gs->gsapi_run_string_end (ghostScriptInstance, 0, &exit_code);
	if (exit_code && !handleExit(exit_code)) return false;
	m_syncDone = false;
	return true;
}

// gs api wrapping
int GSInterpreterLib::gs_input  ( char* /* buffer */, int len )
{
	// emit io (Input,buffer,len);
	return len;
}
int GSInterpreterLib::gs_output ( const char* /* buffer */, int len )
{
	// if (m_buffered) m_buffer.append(QString::fromLocal8Bit (buffer,len));
	// emit io (Output,buffer,len);
	return len;
}

int GSInterpreterLib::gs_error  ( const char* /* buffer */, int len )
{
	// if (m_buffered) m_buffer.append(QString::fromLocal8Bit (buffer,len));
	// emit io (Error,buffer,len);
	return len;
}

int GSInterpreterLib::presize(void * /* device*/, int width, int height, int raster, unsigned int format)
{
	m_Gwidth=width;
	m_Gheight=height;
	m_raster=raster;
	m_format=format;
	return 0;
}

int GSInterpreterLib::size(void * /* device*/, int /* wd */, int /* hi*/, int /*raster*/, unsigned int /*format*/, unsigned char *pimage)
{
	m_wasSize = true;
	m_imageChar = pimage;
	return 0;
}

int GSInterpreterLib::update(void * /* device*/, int x, int y, int w, int h)
{
	if (m_wasSize && !m_wasPage && m_imageChar!=0 && m_img!=0)
	{
		unsigned char *s;
		int th=y+h,tw=x+w;
		for (int i=y;i<th;i++)
		{
			for (int j=x;j<tw;j++)
			{
				s = m_imageChar + (i*m_raster) + j*4;
				uint* p= reinterpret_cast<uint*> (m_img->scanLine(i) + j*4);
				*p=qRgb( (uint) s[0], (uint) s[1], (uint) s[2] );
			}
		}
		QRect t(x,y,w,h);
		emit Updated(t);
	}
	return 0;
}

#include <qdialog.h>
#include <qlabel.h>
#include <qpixmap.h>

int GSInterpreterLib::page(void * /* device*/, int /* copies*/, int /* flush*/)
{
	m_wasPage=true;
	if (m_sync && !m_syncDone)
	{
		// assume the image given by the gs is of the requested widthxheight
		if (! progressive() )
		{
			if (m_cnvmode == QGS_DISPLAY_24) {
				qDebug("Converting 24 bit data");
				int dest = 0, src = 0;
				if (m_conv != NULL) delete[] m_conv;
				m_conv = new unsigned char[m_Gwidth*m_Gheight*4];
				for (int j = 0; j < m_Gheight; j++) {
					src = j * m_raster;
					for (int i = 0; i < m_Gwidth; i++) {
						m_conv[dest++] = m_imageChar[src++];
						m_conv[dest++] = m_imageChar[src++];
						m_conv[dest++] = m_imageChar[src++];
						m_conv[dest++] = 0;
					}
				}
				m_img = new QImage(m_conv, m_Gwidth, m_Gheight, QImage::Format_RGB32);
			} else {
				m_img = new QImage(m_imageChar, m_Gwidth, m_Gheight, QImage::Format_RGB32);
			}
			emit Finished(m_img->copy());
		}
		else
		{
			emit Finished();
		}
		m_syncDone=true;
	}
	return 0;
}

QImage GSInterpreterLib::getImageCopy()
{
	return(m_img->copy());
}

bool GSInterpreterLib::handleExit(int code)
{
	// qWarning("Got code %d",code);
	if ( code>=0 )
		return true;
	else if ( code <= -100 )
	{
		switch (code)
		{
			case e_Fatal:
				m_error.setError(e_Fatal, "fatal internal error");
				return false;
			case e_ExecStackUnderflow:
				m_error.setError(e_ExecStackUnderflow, "stack overflow");
				return false;
			// no error or not important
			case e_Quit:
			case e_Info:
			case e_InterpreterExit:
			case e_RemapColor:
			case e_NeedInput:
			case e_NeedStdin:
			case e_NeedStdout:
			case e_NeedStderr:
			case e_VMreclaim:
			default:
				return true;
		}
	}
	else
	{
		const char* errors[]= { "", ERROR_NAMES };
		m_error.setError(code, errors[-code]);
		return false;
	}
}

void GSInterpreterLib::argsToChar()
{
	if (m_argsChar)
	{
		for (int i=0;i<m_argsCCount;i++) delete [] *(m_argsChar+i);
		delete [] m_argsChar;
	}
	QStringList internalArgs;
	if (m_args.isEmpty())
	{
		internalArgs << " "
		<< "-dMaxBitmap=10000000 "
		<< "-dDELAYSAFER"
		<< "-dNOPAUSE"
		<< "-dNOPAGEPROMPT"
		<< QString("-r%1").arg(m_dpi)
		<< QString("-g%1x%2").arg(m_width).arg(m_height)
		<< QString("-dDisplayResolution=%1").arg(m_dpi)
		<< QString("-dTextAlphaBits=%1").arg(m_textaa)
		<< QString("-dGraphicsAlphaBits=%1").arg(m_graphaa);
		if ( !m_pfonts ) internalArgs << "-dNOPLATFONTS";
	}
	else
	{
		internalArgs = m_args;
	}

	if (m_display)
	{
		QString addr;
		// Use %p to print pointers to avoid problems with conversion to singed integers
		addr.sprintf("%p", this);
		if (addr.startsWith("0x")) addr.remove(0,2);
		int format = DISPLAY_COLORS_RGB | DISPLAY_DEPTH_8 | DISPLAY_LITTLEENDIAN | DISPLAY_TOPFIRST;
		if (m_cnvmode == 0) {
			format |= DISPLAY_UNUSED_LAST;
		}
		internalArgs << QString().sprintf("-dDisplayFormat=%d", format) << QString("-dDisplayHandle=16#%1").arg(addr);
	}

	int t=internalArgs.count();
	char ** args=static_cast <char**> (new char* [t]);
	for (int i=0;i<t;i++)
	{
		*(args+i)=new char [internalArgs[i].length()+1];
		qstrcpy (*(args+i),internalArgs[i].toLocal8Bit());
	}
	m_argsChar=args;
	m_argsCCount=t;
}

bool GSInterpreterLib::hasError()
{
	return m_error.hasError();
}
	
const GSError& GSInterpreterLib::getError()
{
	return m_error;
}

GSError::GSError()
{
	m_code = 0;
	m_haserr = false;
}

GSError::~GSError()
{
}

void GSError::setError(int code, QString msg) 
{
	m_code = code;
	m_name = msg;
	m_haserr = true;
}
