/***********************************************************************************
 * QGLE - A Graphical Interface to GLE                                             *
 * Copyright (C) 2006  A. S. Budden & J. Struyf                                    *
 *                                                                                 *
 * This program is free software; you can redistribute it and/or                   *
 * modify it under the terms of the GNU General Public License                     *
 * as published by the Free Software Foundation; either version 2                  *
 * of the License, or (at your option) any later version.                          *
 *                                                                                 *
 * This program is distributed in the hope that it will be useful,                 *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of                  *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the                   *
 * GNU General Public License for more details.                                    *
 *                                                                                 *
 * You should have received a copy of the GNU General Public License               *
 * along with this program; if not, write to the Free Software                     *
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA. *
 *                                                                                 *
 * Also add information on how to contact you by electronic and paper mail.        *
 ***********************************************************************************/

#ifndef __GLESERVERTHREAD_H
#define __GLESERVERTHREAD_H

#include <QtGui>
#include <QtNetwork>
#include "qgle_definitions.h"
#include "fileinfo.h"
//#include "psfunctions.h"


//! TCP Server Thread definition
/*!
 * This class contains the code to run a TCP server
 * and process the result of connections to that server.
 * It also performs the calls to GS and should send a QImage 
 * to the drawingArea (via a queued signal) to make the on-screen
 * image update
 */
class GLEServerThread : public QThread
{
	Q_OBJECT

public:
	//! Constructor
	/*! As a thread, it should have a parent.  The calling function
	 * can specify a port to listen on
	 */
	GLEServerThread(QObject *parent, quint16 port = DEFAULT_PORT);
	~GLEServerThread();

	//! This starts the thread: run()
	void initialise();

	//! Is the server ready?
	bool isListening();
	
	//! Wait until the server is up and running
	void waitForServer();
	
	//! Is Autoscale of image enabled
	bool shouldAutoScale();

	//! Autoscale the next image
	void setAutoScale(bool ascale);

	//! Return the default resolution
	double defaultResolution();

	//! Enumerator showing how the rendering was requested
	enum
	{
		CalledBySocket,
		CalledByQGLE
	};

signals:
	//! This signal notifies the drawingArea that a new image is ready
	void imageReady(GLEFileInfo image);
	//! This signal keeps the user notified of what's going on.
	void serverMessage(QString msg);
	//! This signal notifies the world that the dpi has changed
	void dpiChanged(double new_dpi);
	//! This signal notifies QGLE that internal image processing is complete
	void readyToGo(QImage image);
	//! This signal notifies QGLE that a new file has been opended using "gle -p"
	void newGleFileSet();

public slots:
	//! Set the resolution
	void setDPI(double new_dpi);
	//! Prepare the file info for transmission to the main thread
	void prepareFileInfo(QImage image);
	//! Convert an EPS file to an image
	void EPSToImage(QString file, double new_dpi, bool autoScale);
	//! What has the user specified for autoscaling?
	void setUserAutoScale(bool ascale);
	//! Set the default resolution
	void setDefaultResolution(double new_dpi);
	

protected:
	//! This is the main processing part of the thread.
	/*!
	 * Starts a TCPServer and when the server receives a new connection it
	 * gets the data from GLE and calls ghostscript.  Finally, it emits
	 * a signal to say that the image is ready.
	 */
	void run();

private:
	//! Set true to quit
	bool abort;
	//! The server object
	QTcpServer *server;
	//! The render function
	void render(int caller);
	//! A mutex for variable protection
	QMutex mutex;
	//! Wait conditions for variable protection
	QWaitCondition condition;
	//! Wait conditions for variable protection
	QWaitCondition serverReady;
	//! A mutex for variable protection
	QMutex serverReadyMutex;
	//! The port to listen on
	quint16 port;
	//! The file info for the rendered file
	GLEFileInfo gleFile;
	//! The resolution
	double dpi;
	//! Is the server listening?
	bool listening;
	bool updateDPI;
	bool userAutoScale;
	//GSInterface *gsInterface;
	double defaultDPI;

};

#endif
