/****************************************************************
 * Licensed to the Apache Software Foundation (ASF) under one   *
 * or more contributor license agreements.  See the NOTICE file *
 * distributed with this work for additional information        *
 * regarding copyright ownership.  The ASF licenses this file   *
 * to you under the Apache License, Version 2.0 (the            *
 * "License"); you may not use this file except in compliance   *
 * with the License.  You may obtain a copy of the License at   *
 *                                                              *
 *   http://www.apache.org/licenses/LICENSE-2.0                 *
 *                                                              *
 * Unless required by applicable law or agreed to in writing,   *
 * software distributed under the License is distributed on an  *
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY       *
 * KIND, either express or implied.  See the License for the    *
 * specific language governing permissions and limitations      *
 * under the License.                                           *
 ****************************************************************/

package org.apache.james.mailrepository.postgres;

import org.apache.james.backends.postgres.PostgresDataDefinition;
import org.apache.james.backends.postgres.PostgresExtension;
import org.apache.james.blob.api.BlobStore;
import org.apache.james.blob.api.TestBlobId;
import org.apache.james.blob.mail.MimeMessageStore;
import org.apache.james.blob.memory.MemoryBlobStoreFactory;
import org.apache.james.mailrepository.MailRepositoryContract;
import org.apache.james.mailrepository.api.MailRepository;
import org.apache.james.mailrepository.api.MailRepositoryPath;
import org.apache.james.mailrepository.api.MailRepositoryUrl;
import org.apache.james.mailrepository.api.Protocol;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.extension.RegisterExtension;

public class PostgresMailRepositoryTest implements MailRepositoryContract {
    static final TestBlobId.Factory BLOB_ID_FACTORY = new TestBlobId.Factory();

    @RegisterExtension
    static PostgresExtension postgresExtension = PostgresExtension.withoutRowLevelSecurity(PostgresDataDefinition.aggregateModules(PostgresMailRepositoryDataDefinition.MODULE));

    private PostgresMailRepository mailRepository;

    @BeforeEach
    void setUp() {
        mailRepository = retrieveRepository(MailRepositoryPath.from("testrepo"));
    }

    @Override
    public MailRepository retrieveRepository() {
        return mailRepository;
    }

    @Override
    public PostgresMailRepository retrieveRepository(MailRepositoryPath path) {
        MailRepositoryUrl url = MailRepositoryUrl.fromPathAndProtocol(new Protocol("postgres"), path);
        BlobStore blobStore = MemoryBlobStoreFactory.builder()
            .blobIdFactory(BLOB_ID_FACTORY)
            .defaultBucketName()
            .passthrough();
        return new PostgresMailRepository(url, new PostgresMailRepositoryContentDAO(postgresExtension.getDefaultPostgresExecutor(), MimeMessageStore.factory(blobStore), BLOB_ID_FACTORY));
    }
}
