/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.myfaces.orchestra.conversation;

import java.io.Serializable;

/**
 * Provide information about the current conversation.
 * <p>
 * An instance of this type is stored in a thread-local variable to indicate
 * what the "current conversation state" is. The getConversation() method
 * can therefore be used to determine what conversation is currently active,
 * and getBeanName() can be used to determine what the most recently-invoked
 * conversation-scoped-bean was. This thread-local variable is maintained
 * via the CurrentConversationAdvice which wraps every conversation-scoped 
 * bean and intercepts all method calls to the bean.  
 * <p>
 * This object also records the fact that a specific bean is within a
 * specific conversation. This data is saved during serialization so
 * that on deserialize we know which conversation to reattach which
 * bean to./
 */
public class CurrentConversationInfo implements Serializable
{
    private final String conversationName;
    private final String beanName;

    /**
     * The conversation object itself is not serializable (it is a Spring proxy
     * object with various AOP advices attached to it). Therefore this reference
     * must be transient. After deserialisation, this member is of course null,
     * but is recalculated on demand in the getConversation method.
     * <p>
     * The beans that are <i>in</i> the conversation are hopefully serializable;
     * they are saved directly and then reattached to the new Conversation instance.
     */
    private transient Conversation conversation;

    public CurrentConversationInfo(Conversation conversation, String beanName)
    {
        this.conversation = conversation;
        this.conversationName = conversation.getName();
        this.beanName = beanName;
    }

    /**
     * The conversation the bean is associated with.
     */
    public Conversation getConversation()
    {
        if(conversation == null)
        {
            ConversationManager conversationManager = ConversationManager.getInstance();
            conversation = conversationManager.getConversation(conversationName);
        }
        return conversation;
    }

    /**
     * The bean name.
     */
    public String getBeanName()
    {
        return beanName;
    }
}
